% This part of the code does the postprocessing.
% We are keeping this separate because it is more time consuming than
% calculating the equilibrium and one may not want all the data every time
% an equilibrium is calculated.
% This version uses surfaces calculated in (z,y).

global eps epsbar twoD psimax nlevels vlevel alphasol eq_option

tic

alphasol
eps
gamma

if(eq_option==1)
    delta
    kappa
elseif(eq_option==2)
    delta_X
    kappa_X
elseif(eq_option==3)
    delta
    kappa
    delta_X
    kappa_X
end



pick_q0 = 1;

safe_small = 5*10.^-3;

twoD = 0;
psimax = 1;

fun_for_max = @(xx) -psi_any_shape(xx(1),xx(2));

xxstart = [0 0];
xxmax = fminsearch(fun_for_max,xxstart);
psimax = abs(psi_any_shape(xxmax(1),xxmax(2)));

zaxis = xxmax(1);
yaxis = xxmax(2);
R_axis = sqrt(2*eps*zaxis + 1 + eps^2) * R0;
Z_axis = a*yaxis;

% Define remaining input parameters (by default, we have already defined R0
% elsewhere).

B0 = 1; % Vacuum toroidal field on geometric axis
if(pick_q0==0)
    
    beta0 = 0.0605 % Toroidal beta on the magnetic axis
    
elseif(pick_q0==1)
    
    q0 = 1;
    psi_xx_0 = psi_xx_any_shape(zaxis,yaxis);
    psi_yy_0 = psi_yy_any_shape(zaxis,yaxis);
    xloc = 1+eps^2+2*eps*zaxis;
    
    psiloc = 1;
    beta0 = gamma*eps^2*alphasol^2/(q0^2*xloc^2*psi_xx_0*psi_yy_0-...
        (1+eps^2)*(1-gamma)*eps^2*alphasol^2)
    
end

mu0  = 4*pi*10^-7;

zmin = -1;
zmax = 1;

if((eq_option==1)||(eq_option==2))
    % Miller shape and Double Null
    ymin = -yTop;
    ymax = yTop;
elseif(eq_option==3)
    % Singe Null
    ymin = yX; % yX<0 by definition
    ymax = yTop;
end


% Calculate algebraic quantities.

p0 = beta0 * B0^2/(2*mu0); % Plasma pressure on axis

dB_ov_B = beta0*(1+eps^2)/2*((1-gamma)/gamma); % Plasma diamagnetism delta B / B0

psi_axis = eps*B0*R0^2/alphasol*sqrt(beta0/gamma); % Magnetic flux on axis

% Plot normalized p and J_phi

zofRfun = @(R) (R.^2./R0^2 - 1- eps^2)./(2*eps);
yofZfun = @(Z) Z/a;
Jphifunb = @(R,Z) (2*mu0*p0*R.^2 + 2*R0^2*B0^2*dB_ov_B).* ...
    psi_any_shape(zofRfun(R),yofZfun(Z))./psi_axis./R;

bigR1D = linspace(R0*(1-eps),R0*(1+eps),nR);
z_for_plot_1D = (bigR1D.^2/R0^2-1-eps^2)/(2*eps);
psi1D = psi_any_shape(z_for_plot_1D,zeros(size(bigR1D)));

p1D = psi1D.^2;
p1D = p1D./max(p1D);
Jphi1D = Jphifunb(bigR1D,zeros(size(bigR1D)))./mu0;
Jphi1D = abs(Jphi1D)./max(abs(Jphi1D));

figure
plot(bigR1D,p1D,'LineWidth',1.5)
hold on
plot(bigR1D,Jphi1D,'LineWidth',1.5)
legend({'Pressure','J_\phi'},'FontSize',14)
legend('Location','northwest')
axis tight





% Now calculate the various integrals that enter into all other quantities.

% First define any needed functions and function handles

% This is used for the area.
fun_Jac2D = @(z) a^2./(sqrt(1+eps^2+2*eps*z));

% We are going to need the plasma shape for various reasons, so we might as
% well determine it here and use it to calculate the surface and line integrals.

ntheta0 = 500; % Number of points on the surface
ntemp = 200;

% Define the input plasma shape.
% Use the desired plasma shape as a starting guess for the outermost
% surface.
if(eq_option==1)
    
    zsfun = @(theta) cos(theta+dh*sin(theta)) - ...
        eps/2 * sin(theta+dh*sin(theta)).^2;
    ysfun = @(theta) kappa*sin(theta);
    
    % Create two y(z) functions that we'll need later.
    % First, positive y.
    
    thetatemp = linspace(0, pi, ntemp);
    ztop_list = zsfun(thetatemp);
    ytop_list = ysfun(thetatemp);
    % Negative y is very similar.
    thetatemp = linspace(pi, 2*pi, ntemp);
    zbot_list = zsfun(thetatemp);
    ybot_list = ysfun(thetatemp);
    
elseif(eq_option==2)
    
    thetaL = linspace(pi-theta0,pi+theta0,ntemp);
    thetaR = linspace(-theta0,theta0,ntemp);
    
    bigX_L = xx1+(1+xx1)*cos(thetaL);
    bigY_L = kappa_0*sin(thetaL);
    
    bigX_R = -xx2+(1+xx2)*cos(thetaR);
    bigY_R = kappa_0*sin(thetaR);
    
    bigX=[bigX_L, bigX_R];
    bigY=[bigY_L, bigY_R];
    
    bigR_edge = a*bigX+R0;
    bigZ_edge = a*bigY;
    
    z_edge = (bigR_edge.^2/R0^2-1-eps^2)/(2*eps);
    y_edge = bigY;
    
    [zboundary_max izmax]=max(z_edge);
    [zboundary_min izmin]=min(z_edge);
    
    zbot_list_step = z_edge(izmin+1:izmax);
    ybot_list_step = bigY(izmin+1:izmax);
    
    [zbot_list isort0] = unique(zbot_list_step);
    ybot_list = ybot_list_step(isort0);
    
    ztop_list_step = [z_edge(1:izmin-1), z_edge(izmax+1:end-1)];
    ytop_list_step = [y_edge(1:izmin-1), y_edge(izmax+1:end-1)];
    
    [ztop_list_step2 isort] = sort(ztop_list_step);
    ytop_list_step2 = ytop_list_step(isort);
    
    [ztop_list isort2] = unique(ztop_list_step2);
    ytop_list = ytop_list_step2(isort2);
    
    
elseif(eq_option==3)
    
    thetaup = linspace(0,pi,ntemp);
    thetaL = linspace(pi,pi+theta0,ntemp);
    thetaR = linspace(2*pi-theta0,2*pi,ntemp);
    
    bigX_up = cos(thetaup+dh*sin(thetaup));
    bigY_up = kappa*sin(thetaup);
    
    bigX_L = xx1+(1+xx1)*cos(thetaL);
    bigY_L = kappa_0*sin(thetaL);
    
    bigX_R = -xx2+(1+xx2)*cos(thetaR);
    bigY_R = kappa_0*sin(thetaR);
    
    bigX=[bigX_up, bigX_L, bigX_R];
    bigY=[bigY_up, bigY_L, bigY_R];
    
    bigR_edge = a*bigX+R0;
    bigZ_edge = a*bigY;
    
    z_edge = (bigR_edge.^2/R0^2-1-eps^2)/(2*eps);
    y_edge = bigY;
    
    ztop_list = z_edge(1:ntemp);
    ytop_list = bigY(1:ntemp);
    
    zbot_list = z_edge(ntemp+1:end);
    ybot_list = bigY(ntemp+1:end);
    
end

fysofzT = spline(ztop_list,ytop_list);
funysofzT = @(z) ppval(fysofzT,z);

fysofzB = spline(zbot_list,ybot_list);
funysofzB = @(z) ppval(fysofzB,z);

% Create the arrays.
npoints = 250;
zlist = NaN(npoints,1);
yuplist = zlist;
ydownlist = zlist;

% Set up known edge points.
zlist = linspace(-1,1,npoints);
yuplist(1) = 0;
yuplist(end) = 0;
ydownlist(1) = 0;
ydownlist(end) = 0;

% Work on the top shape first.
for i = 2:npoints-1
    
    zloc = zlist(i);
    ffory = @(y) psi_any_shape(zloc,y);
    zerostartb = yuplist(i-1);
    zerostart = [0 zerostartb];
    while(ffory(zerostart(1,1))*ffory(zerostart(1,2))>0)
        temp = zerostart(1,2) + 0.002;
        zerostart(1,2) = temp;
        
    end
    yloc = fzero(ffory,zerostart);
    yuplist(i) = yloc;
    
end

% Repeat the calculation for the bottom shape.
for i = 2:npoints-1
    
    zloc = zlist(i);
    ffory = @(y) psi_any_shape(zloc,y);
    zerostartb = ydownlist(i-1);
    zerostart = [0 zerostartb];
    while(ffory(zerostart(1,1))*ffory(zerostart(1,2))>0)
        temp = zerostart(1,2) - 0.002;
        zerostart(1,2) = temp;
        
    end
    yloc = fzero(ffory,zerostart);
    ydownlist(i) = yloc;
    
end

zlist_for_FLOW = [flipud(zlist'); zlist(2:end)'];
ylist_for_FLOW = [flipud(yuplist); ydownlist(2:end)];

R_for_FLOW = R0 * sqrt(1 + eps^2 + 2*eps*zlist_for_FLOW);
Z_for_FLOW = a*ylist_for_FLOW;

save_for_FLOW = [R_for_FLOW Z_for_FLOW];

save('R_Z_shape.dat','save_for_FLOW','-ascii');

fysofzT = spline(zlist,yuplist);
funysofzT = @(z) ppval(fysofzT,z);

fysofzB = spline(zlist,ydownlist);
funysofzB = @(z) ppval(fysofzB,z);

ytoplast = @(z) funysofzT(z);
ybotlast = @(z) funysofzB(z);

% Notice that we calculate the same integrals twice using different
% routines and methods for debugging purposes. This is overkill for this
% problem. The older version can be commented out.

fun_A_new = @(z,y) fun_Jac2D(z);
fun_A_new = @(z,y) 1+z-z;

fun_psi2_new = @(z,y) psi_any_shape(z,y).^2;
fun_psi2_new = @(z,y) psi_for_integral(z,y).^2;

fun_t1_new = @(z,y) (1+gamma*epsbar*z)./(1+epsbar*z).* psi_for_integral(z,y);

fun_t2_new = @(z,y) (1+gamma*epsbar*z)./(1+epsbar*z).* fun_psi2_new(z,y);

% Now get the integrals

int_A_new = integral2(fun_A_new,zmin,zmax,ybotlast,ytoplast)

int_psi2_new = integral2(fun_psi2_new,zmin,zmax,ybotlast,ytoplast);

int_t2_new = integral2(fun_t2_new,zmin,zmax,ybotlast,ytoplast);

int_t1_new = integral2(fun_t1_new,zmin,zmax,ybotlast,ytoplast);


fun_A = @(z,y) fun_Jac2D(z).*psi_for_integral(z,y)./(psi_for_integral(z,y)+10^-16);

fun_psi2 = @(z,y) psi_for_integral(z,y).^2;

fun_t1 = @(z,y) (1+gamma*epsbar*z)./(1+epsbar*z).* psi_for_integral(z,y);

fun_t2 = @(z,y) (1+gamma*epsbar*z)./(1+epsbar*z).* fun_psi2(z,y);

% Now get the integrals

int_A = integral2(fun_A,zmin,zmax,ymin,ymax,'AbsTol',1e-2,'RelTol',1e-4);

int_psi2 = integral2(fun_psi2,zmin,zmax,ymin,ymax);

int_t2 = integral2(fun_t2,zmin,zmax,ymin,ymax);

int_t1 = integral2(fun_t1,zmin,zmax,ymin,ymax);

% Now we can calculate the desired integral quantities.

% betator = beta0 * int_psi2/int_A;
betator_new = beta0 * int_psi2_new/int_A_new

% betapol = gamma * int_psi2/int_t2;
betapol_new = gamma * int_psi2_new/int_t2_new

I_plasma = eps*B0*R0*alphasol/mu0 * sqrt(beta0/gamma) * int_t1;
I_plasma_new = eps*B0*R0*alphasol/mu0 * sqrt(beta0/gamma) * int_t1_new

l_inductance = 4*pi/alphasol^2 * int_t2/int_t1^2;
l_inductance_new = 4*pi/alphasol^2 * int_t2_new/int_t1_new^2


% We have moved q_star to the end because it requires kappa_95 for the
% cases with X points

% We need to explicitly define F(psi) for the FLOW version.
bigF = @(psi) R0*B0 *sqrt(1 + 2*dB_ov_B *psi.^2);

% q_star_2 = 2*int_A*bigF(0)/(mu0*R0^2*I_plasma);
% q_star_2_new = 2*int_A_new*bigF(0)/(mu0*R0^2*I_plasma);


% Now we move to line integrals. These only matter for q(psi).
% It is easier to work on one contour at a time, so add here any other line
% integrals that may come up.

% We use Jeff's simplification of the integrand to
% ymax(psi)*sin(theta)/psi_z(z(theta),y(theta))

% Set a psi value for each surface. Use the psi value to determine the z
% values for the y=0 axis first, then set up a grid in z and determine y.

enq = 10;
ntheta = 401*ones(enq,1); % Edit into a non-constant array if needed. Any odd number will work.
qtab_smooth = NaN(enq+1,2); % Includes the axis

psi_for_q_list = NaN(enq,1);
psi_for_q_list = linspace(0,enq/(enq+1),enq);
%psi_for_q_list(end) = 0.999; % Uncomment this line to confirm q0.

% Make sure that one value is 0.05, so that we can give q95
[temppsi, i95] = min(abs(psi_for_q_list(2:end)-0.05));
psi_for_q_list(i95+1) = 0.05;

zlastL = -1;
zlastR = 1;

% The safety factor goes to infinity on the LCFS if there is an X point.
% For that reason, we start from the second value of psi.
% If so desired, one can run the integrals starting from the psi=0 surface
% for all shapes. The code will return a "large" value if there is an
% X-point.
if(eq_option==1)
    iqstart = 1;
else
    iqstart = 2;
end

for i = iqstart:enq
    % Start from the outside and move inwards.
    
    psiloc = psi_for_q_list(i);
    nthetaloc = ntheta(i);
    nthetamid = (nthetaloc-1)/2;
    
    % Note that this overwrites the previous lists.
    zlist = NaN(nthetaloc,1);
    ylist = NaN(nthetaloc,1);
    % Find the left and right crossings of the horizontal axis for this
    % value of psi.
    
    if(i==1)
        % We set up the arrays in a different way following earlier versions.
        % The easiest thing to do is to redefine the arrays in the same way as
        % for all other surfaces.
        zcrossL = -1;
        zcrossR = 1;
        
        zlastL = zcrossL;
        zlastR = zcrossR;
        
        % Fill the z grid
        zlist(1:nthetamid) = linspace(zcrossR,zcrossL,nthetamid);
        zlist(nthetamid:end) = linspace(zcrossL,zcrossR,nthetamid+2);
        
        % Fill the y grid using the functions created earlier.
        % (Repeated point is just to agree with the rest, it does not create any
        % issues)
        ylist(1:nthetamid) = ytoplast(zlist(1:nthetamid));
        ylist(nthetamid:end) = ybotlast(zlist(nthetamid:end));
        
    else
        
        fforcross = @(z) psi_any_shape(z,0)-psiloc;
        crossstart = [zlastL zaxis];
        zcrossL = fzero(fforcross,crossstart);
        crossstart = [ zaxis zlastR];
        zcrossR = fzero(fforcross,crossstart);
        
        zlastL = zcrossL;
        zlastR = zcrossR;
        
        % Fill the z grid
        zlist(1:nthetamid) = linspace(zcrossR,zcrossL,nthetamid);
        zlist(nthetamid:end) = linspace(zcrossL,zcrossR,nthetamid+2);
        
        % Now fill the y grid. We need to do this by steps.
        
        ylist(1) = 0;
        
        for j = 2:nthetamid-1
            
            zloc = zlist(j);
            ffory = @(y) psi_any_shape(zloc,y)-psiloc;
            zerostartb = ytoplast(zloc);
            zerostart = [0 zerostartb];
            jj = 0;
            while(ffory(zerostart(1,1))*ffory(zerostart(1,2))>0)
                temp = zerostart(1,2) + 0.002;
                zerostart(1,2) = temp;
            end
            yloc = fzero(ffory,zerostart);
            ylist(j) = yloc;
            
        end
        
        ylist(nthetamid) = 0;
        
        for j = nthetamid+1:nthetaloc-1
            
            zloc = zlist(j);
            ffory = @(y) psi_any_shape(zloc,y)-psiloc;
            zerostart = [ybotlast(zloc) 0];
            while(ffory(zerostart(1))*ffory(zerostart(2))>0)
                zerostart(1) = zerostart(1)-0.002;
            end
            yloc = fzero(ffory,zerostart);
            ylist(j) = yloc;
            
        end
        
        ylist(nthetaloc) = 0;
        
        % Now redefine the y(z) functions and use them to find ymax(s) for this
        % surface.
        
        fystepofzB = spline(zlist(1:nthetamid),ylist(1:nthetamid));
        ytoplast = @(z) ppval(fystepofzB,z);
        
        fystepofzT = spline(zlist(nthetamid:end),ylist(nthetamid:end));
        ybotlast = @(z) ppval(fystepofzT,z);
        
    end
    yuplist = ylist(1:nthetamid);
    ydownlist = ylist(nthetamid+1:end);
    
    [zbot, ybot] = fminbnd(ybotlast,zlastL,zlastR);
    ymaxB = -ybot;
    
    ytemp = @(z) -ytoplast(z);
    [ztop, ytop] = fminbnd(ytemp,zlastL,zlastR);
    ymaxT = -ytop;
    
    if(psiloc==0.05)
        kappa95 = (ymaxB+ymaxT)/2;
    end
    
    % Now determine theta
    
    thetatempT = real(asin(yuplist./ymaxT));
    thetatempB = real(asin(ydownlist./ymaxB));
    
    [thmaxT iTop]=max(thetatempT);
    thetatempT(iTop+1:end) = pi - thetatempT(iTop+1:end);
    
    [thminB iBot]=min(thetatempB);
    thetatempB(1:iBot) = -thetatempB(1:iBot)+pi;
    thetatempB(iBot+1:end) = 2*pi + thetatempB(iBot+1:end);
    
    theta_list = [thetatempT; thetatempB];
    
    
    spline_z_level = spline(theta_list,zlist);
    zlvlfun = @(x) ppval(spline_z_level,x);
    
    % We need two integrands because the plasma shape may not be symmetric.
    fq_smoothT = @(th) -ymaxT*cos(th)./((1+eps^2+2*eps*zlvlfun(th)) .* ...
        psi_x_any_shape(zlvlfun(th),ymaxT*sin(th)));
    fq_smoothB = @(th) -ymaxB*cos(th)./((1+eps^2+2*eps*zlvlfun(th)) .* ...
        psi_x_any_shape(zlvlfun(th),ymaxB*sin(th)));
    
    % Now, the issue is that this function is analytically smooth, but
    % numerically it can blow up near the zeros of the denominator. As of
    % now, it is unclear why this happens, but we'll deal with it as we can
    % for a start.
    
    % First, find the zeros.
    
    fden = @(th) psi_x_any_shape(zlvlfun(th),ymaxT*sin(th));
    zero1 = fzero(fden,pi/2);
    fden = @(th) psi_x_any_shape(zlvlfun(th),ymaxB*sin(th));
    zero2 = fzero(fden,3*pi/2);
    
    % Then break the integral in four parts, just "dropping" the trouble
    % spots.
    
    % Part 1
    
    th1_end = zero1-safe_small;
    int_part1 = integral(fq_smoothT,0,th1_end);
    th_interval_1 = th1_end;
    th_interval_1_right = zero1;
    
    int_part1 = int_part1/th_interval_1*th_interval_1_right;
    
    % Part 2
    th2_start = zero1+safe_small;
    th2_end = pi;
    int_part2 = integral(fq_smoothT,th2_start,th2_end);
    th_interval_2 = th2_end-th2_start;
    th_interval_2_right = pi-zero1;
    
    int_part2 = int_part2/th_interval_2*th_interval_2_right;
    
    % Part 3
    th3_start = pi;
    th3_end = zero2-safe_small;
    int_part3 = integral(fq_smoothB,th3_start,th3_end);
    th_interval_3 = th3_end-th3_start;
    th_interval_3_right = th_interval_3+safe_small;
    
    int_part3 = int_part3/th_interval_3*th_interval_3_right;
    
    
    % Part 4
    th4_start = zero2+safe_small;
    th4_end = 2*pi;
    int_part4 = integral(fq_smoothB,th4_start,th4_end);
    th_interval_4 = th4_end-th4_start;
    th_interval_4_right = th_interval_4+safe_small;
    
    int_part4 = int_part4/th_interval_4*th_interval_4_right;
    
    inte_smooth = int_part1+int_part2+int_part3+int_part4;
    
    qloc_smooth = inte_smooth * bigF(psiloc)/(R0*B0)*sqrt(gamma/beta0)* ...
        eps*alphasol/(2*pi);
    
    qtab_smooth(i,:) = [psiloc qloc_smooth];
    if(psiloc==0)
        qa = qloc_smooth
    elseif(psiloc==0.05)
        q95 = qloc_smooth
    end
    
end

% q_axis calculation

if(pick_q0==0)
    
    psi_xx_0 = psi_xx_any_shape(zaxis,yaxis);
    psi_yy_0 = psi_yy_any_shape(zaxis,yaxis);
    xloc = 1+eps^2+2*eps*zaxis;
    
    psiloc = 1;
    q0 = bigF(psiloc)/(R0*B0) * sqrt(gamma/beta0) * eps * alphasol / ...
        (xloc * sqrt(psi_xx_0*psi_yy_0))
    
else
    
    psiloc=1;
    q0
    
end

qtab_smooth(end,:) = [psiloc q0];

% End of q calculation
% Keep in mind that we are only using 50 points for the level curves, but
% one is free to use as many as desired. Since the solution is analytical,
% there is no intrinsic limitation on the resolution!

% Now also calculate q_star
if(eq_option==1)
    q_star = pi*eps/alphasol * sqrt(gamma/beta0) * (1+kappa^2)/int_t1;
    q_star_new = pi*eps/alphasol * sqrt(gamma/beta0) * (1+kappa^2)/int_t1_new
elseif(eq_option==2)
    q_star = pi*eps/alphasol * sqrt(gamma/beta0) * (1+kappa95^2)/int_t1;
    q_star_new = pi*eps/alphasol * sqrt(gamma/beta0) * (1+kappa95^2)/int_t1_new
elseif(eq_option==3)
    q_star = pi*eps/alphasol * sqrt(gamma/beta0) * (1+kappa95^2)/int_t1;
    q_star_new = pi*eps/alphasol * sqrt(gamma/beta0) * (1+kappa95^2)/int_t1_new
end

% Save some more FLOW input.

psilist = linspace(0,1,40);
plist = beta0*B0^2/(2*mu0)*psilist.^2;
Flist = bigF(psilist);

pofpsi_out = [psilist; plist]';
Fofpsi_out = [psilist; Flist]';

save('p_iso.dat','pofpsi_out','-ascii');
save('b0.dat','Fofpsi_out','-ascii');

save('R_Z_shape.dat','save_for_FLOW','-ascii');

if(noplots==0)
    figure
    plot(qtab_smooth(iqstart:end,1),qtab_smooth(iqstart:end,2),'-o')
    hold on
end

toc

% Other checks. There are for debugging purposes and comparison with FLOW.

% small = 10.^-6;
% zleft = zaxis-small;
% zright = zaxis+small;
% yleft = yaxis-small;
% yright = yaxis+small;
%
% psi_xx_0_test = (psi_for_integral(zleft,yaxis) - ...
%     2*psi_for_integral(zaxis,yaxis) + ...
%     psi_for_integral(zright,yaxis))/small^2;
%
% psi_yy_0_test = (psi_for_integral(zaxis,yleft) - ...
%     2*psi_for_integral(zaxis,yaxis) + ...
%     psi_for_integral(zaxis,yright))/small^2;
%
% (psi_xx_0_test-psi_xx_0)/psi_xx_0
%
% (psi_yy_0_test-psi_yy_0)/psi_yy_0

% qofpsi = load('./temp/qofpsi_11.dat');
% qofpsi(:,1) = qofpsi(:,1)/qofpsi(1,1);
% plot(qofpsi(:,1),qofpsi(:,2),'-o')
