module vacuum_module

	use mod_parameters, only : dkind
	use mod_arrays, only : eta_phys

	implicit none

!	private

!	public set_vacuum, nil_vacuum, res_eta, res_factor, res_eta_max, res_eta_min, res_eta_xjump,  &
!				resistive_diffusion, edge_distance, vacuum_input

	integer :: theta_points, resistivity_points
	integer, parameter :: r_ord = 3
	integer, parameter :: res_ord = 2

	real(kind=dkind), dimension(:,:), allocatable :: r_data
	real(kind=dkind), dimension(:,:), allocatable :: r_cscoef
	! radius coordinates and interpolation coefficients

	real(kind=dkind), dimension(:,:), allocatable :: res_data
	real(kind=dkind), dimension(:,:), allocatable :: res_cscoef
	! resistivity "coordinates" and interpolation coefficients

	real(kind=dkind), dimension(:,:,:), allocatable :: res_eta
	! resistivity and its derivatives:
	! _0 -> resistivity, _1/2 -> x/y derivatives

	real(kind=dkind), dimension(:,:), allocatable :: res_eta_iph, res_eta_jph
	! resistivity in half-grid points

	real(kind=dkind), dimension(:,:), allocatable :: edge_distance

	real(kind=dkind) :: res_factor	=15.d0	! factor in tanh
	real(kind=dkind) :: res_eta_max = 1.d0
	real(kind=dkind) :: res_eta_min	= 1.d-6
	! maximum and minimum of resistivity

	real(kind=dkind) :: res_eta_xjump	! position where the resistivity increases

	integer :: res_implicit_option
	! 0 -> only del**2 is treated implicitly
	! 1 -> del**2 and all terms multiplying eta are treated implicitly
	! 2 -> everything (including terms with eta derivatives) is treated implicitly
	! 3 -> everything treated together, with direct discretization of the current curl
	! 4 -> as in 2, but eta~T**(-res_phys_exp)
	! 5 -> as in 2, but eta~rho**(-res_phys_exp)
	! 6 -> as in 2, but eta~p**(-res_phys_exp)
	! 14-16 -> as 4-6, but the resistivity is limited to eta<=res_eta_max
	! 25 -> as 15, but resistivity = numerical function of rho
	! 35 -> as 15, but resistivity = analytical function of rho
	! 45 -> as 15, but resistivity = analytical function of fictitious rho

	real(kind=dkind) :: res_phys_exp
	! for calculating resistivity profile based on temperature or density

	real(kind=dkind) :: R_P, y_P	! used to determine point distance from the boundary

	real(kind=dkind) :: res_xmin, res_xmax
	real(kind=dkind) :: res_scale = 0.d0

	real(kind=dkind) :: eta_phys_boundary, eta_phys_center
	logical :: initialize_eta_phys

        integer :: additional_res_steps = 0

	namelist/vacuum_input/ res_eta_max, res_eta_min, res_factor, res_eta_xjump, res_implicit_option, res_phys_exp,  &
												eta_phys_boundary, eta_phys_center, initialize_eta_phys, additional_res_steps


	contains

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine nil_vacuum
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! this routine sets up the appropriate zeros in case of no vacuum

	call res_allocation

	res_eta = 0.d0

end subroutine nil_vacuum

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine set_vacuum
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! this routine sets up the vacuum, taking care of two tasks:
! 1) -> load the plasma geometry
! 2) -> calculate the resistivity profiles

	real(kind=dkind), save :: set_vacuum_switch = 0.d0

	if(set_vacuum_switch==0.d0) then

		call res_allocation
		res_eta = res_eta_max

		call setup_radius

		if(res_implicit_option==25)	 call setup_resistivity_numerical

		if((res_implicit_option==4).or.(res_implicit_option==5).or.(res_implicit_option==6).or.  &
			(res_implicit_option==14).or.(res_implicit_option==15).or.(res_implicit_option==16) &
			.or.(res_implicit_option==25).or.(res_implicit_option==35).or.(res_implicit_option==45)) then
			continue
		else
			call setup_resistivity
		endif

		if(res_implicit_option==3) call setup_resistivity_half_points

		set_vacuum_switch = 1.d0

	elseif(set_vacuum_switch==1.d0) then

		call setup_resistivity

	endif

end subroutine set_vacuum

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine res_allocation
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	use mod_parameters, only : nx, ny

	allocate(res_eta(0:nx+1,0:ny+1,0:2))

	if(res_implicit_option==3) 	then
		allocate(res_eta_iph(0:nx+1,0:ny+1))
		allocate(res_eta_jph(0:nx+1,0:ny+1))
	endif


end subroutine res_allocation

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  
subroutine setup_radius
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  

	use pseudo_IMSL, only : DBSNAK, DBSINT

	integer :: i
	integer, parameter :: points_max = 2000
	real(kind=dkind), dimension(points_max,2) :: dummy

	i = 0

	open(33,file='./input/r_of_theta.dat', status='old', action='read')

	do 

		i = i+1
		read(33,*,end=99) dummy(i,1),dummy(i,2)

	enddo

99	continue

	close(33)

	theta_points = i-1

	allocate(r_data(theta_points+r_ord,3))
	allocate(r_cscoef(1,theta_points))

	do i=1,theta_points

		r_data(i,1) = dummy(i,1)
		r_data(i,2) = dummy(i,2)

	enddo


	call DBSNAK(theta_points, r_data(1:theta_points,1),  &
					r_ord,r_data(:,3))

	call DBSINT(theta_points, r_data(1:theta_points,1),  &
		 r_data(1:theta_points,2), r_ord,  &
		 r_data(:,3),  &
		 r_cscoef(1,1:theta_points))

end subroutine setup_radius


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  
subroutine setup_resistivity_numerical
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  

	use pseudo_IMSL, only : DBSNAK, DBSINT

	integer :: i
	integer, parameter :: points_max = 2000
	real(kind=dkind), dimension(points_max,2) :: dummy

	i = 0

	open(33,file='./input/resistivity.dat', status='old', action='read')

	do 

		i = i+1
		read(33,*,end=99) dummy(i,1),dummy(i,2)

	enddo

99	continue

	close(33)

	resistivity_points = i-1

	allocate(res_data(resistivity_points+res_ord,3))
	allocate(res_cscoef(1,resistivity_points))

	do i=1,resistivity_points

		res_data(i,1) = dummy(i,1)
		res_data(i,2) = dummy(i,2)

	enddo


	call DBSNAK(resistivity_points, res_data(1:resistivity_points,1),  &
					res_ord,res_data(:,3))

	call DBSINT(resistivity_points, res_data(1:resistivity_points,1),  &
		 res_data(1:resistivity_points,2), res_ord,  &
		 res_data(:,3),  &
		 res_cscoef(1,1:resistivity_points))

end subroutine setup_resistivity_numerical


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine setup_resistivity
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! calculate the resistivity as function of the position

	use mod_parameters, only : nx, ny
	use mod_arrays, only : xV, yV

	real(kind=dkind) :: dist, dir, angle
	real(kind=dkind) :: res, dres
	integer :: i, j

	allocate(edge_distance(0:nx+1,0:ny+1))
	! also save the distance from the plasma surface, for momentum source profiles

	edge_distance = 1.d1

	do j = 0, ny+1
	do i = 0, nx+1

		call get_dist(i,j,dist,dir, angle)

		dist = dist * dir

		if(res_implicit_option<4) call resistivity_vals(dist,res,dres)

		res_eta(i,j,0) = res
		res_eta(i,j,1) = dres * cos(angle) * dir
		res_eta(i,j,2) = dres * sin(angle) * dir

		edge_distance(i,j) = dist

	enddo
	enddo

	if((res_implicit_option==4).or.(res_implicit_option==5).or.(res_implicit_option==6).or.  &
			(res_implicit_option==14).or.(res_implicit_option==15).or.(res_implicit_option==16)  &
			.or.(res_implicit_option==25).or.(res_implicit_option==35).or.(res_implicit_option==45)) call resistivity_physical

	! save only for debugging

!!$	open(69,file='resistivity.plt')
!!$	write(69,*)'TITLE="resistivity"'
!!$	write(69,*) 'Variables = "X", "Y","eta","eta_x","eta_y"'
!!$	write(69,*)'ZONE I=',nx+2,',J=',ny+2,',F=Point'
!!$
!!$	do  j=0,ny+1
!!$	do  i=0,nx+1
!!$
!!$		write(69,*) xV(i), yV(j), res_eta(i,j,0), res_eta(i,j,1), res_eta(i,j,2)
!!$
!!$	enddo
!!$	enddo
!!$
!!$	close(69)
!!$
!!$	open(69,file='distance.plt')
!!$	write(69,*)'TITLE="distance"'
!!$	write(69,*) 'Variables = "X", "Y","distance"'
!!$	write(69,*)'ZONE I=',nx,',J=',ny,',F=Point'
!!$
!!$	do  j=1,ny
!!$	do  i=1,nx
!!$
!!$		write(69,*) xV(i), yV(j), edge_distance(i,j)
!!$
!!$	enddo
!!$	enddo
!!$
!!$	close(69)

end subroutine setup_resistivity

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine setup_resistivity_half_points
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! calculate the resistivity as function of the position in half-grid points
! this routine is essentially identical to the previous one, but coordinates are shifted
! (and then restored to the original values)

	use mod_parameters, only : nx, ny, dx, dy
	use mod_arrays, only : xV, yV

	real(kind=dkind) :: dist, dir, angle
	real(kind=dkind) :: res, dres
	integer :: i, j

	! do (i+1/2,j) points first

	do i = 0, nx+1
		xV(i) = xV(i) + dx/2
	enddo

	do j = 1, ny
	do i = 0, nx+1

		call get_dist(i,j,dist,dir, angle)

		dist = dist * dir

		call resistivity_vals(dist,res,dres)

		res_eta_iph(i,j) = res

	enddo
	enddo

	! don't forget to restore the original grid points!
	do i = 0, nx+1
		xV(i) = xV(i) - dx/2
	enddo

	! second, do (i,j+1/2) points

	do j = 0, ny+1
		yV(i) = yV(i) + dy/2
	enddo

	do j = 0, ny+1
	do i = 1, nx

		call get_dist(i,j,dist,dir, angle)

		dist = dist * dir

		call resistivity_vals(dist,res,dres)

		res_eta_jph(i,j) = res

	enddo
	enddo

	! don't forget to restore the original grid points!
	do j = 0, ny+1
		yV(i) = yV(i) - dy/2
	enddo



end subroutine setup_resistivity_half_points


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine get_dist(iloc,jloc,dist,dir, angle)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! calculate the distance from the boundary

	use mod_parameters, only : rmajor, xlength, ylength, pi
	use mod_arrays, only : xV, yV

	real(kind=dkind) :: dist, dir
	integer :: iloc, jloc
	real(kind=dkind) :: angle
	real(kind=dkind) :: ex, ey, r, rprim, rsec, theta
	real(kind=dkind) :: xvec(2)
	integer :: ntrial = 50000
	real(kind=dkind) :: tolx=1.d-12
	real(kind=dkind) :: tolf = 1.d-9
	logical :: newton_check
	real(kind=dkind), dimension(1:3) :: theta_start
	real(kind=dkind) :: thetamin
	integer :: t, tmin
	real(kind=dkind) :: dist_min

	R_P = xV(iloc) + rmajor - xlength/2.d0
	y_P = yV(jloc) - ylength/2.d0

!	call radius(iloc,jloc,ex,ey,r)
	call radius_der(R_P,y_P,ex,ey,theta,r,rprim,rsec)

	if(ex**2+ey**2>r**2) then
		dir = 1.d0	! external point
	else
		dir = -1.d0	! internal point
	endif

	! patch for obscure bug: use minimization near the plasma center
	if((abs(xV(iloc)-xlength/2.d0)<xlength/3.d0).and.(abs(yV(jloc)-ylength/2.d0)<ylength/3.d0)) then
	! the point is close to the center

		dist_min = 1.d10

		do t = 1, 201

			theta = (t-1.d0)/200.d0 * 2.d0*pi
			dist = dist_fun(theta)
			if(dist<dist_min) then
				theta_start(2) = theta
				dist_min = dist
			endif

		enddo

		theta_start(1) = theta_start(2) - pi/100.d0
		theta_start(3) = theta_start(2) + pi/100.d0

		dist = brent(theta_start(1), theta_start(2), theta_start(3), dist_fun, tolf, thetamin)

		call radius_theta(thetamin,r,xvec(1),xvec(2))

	else

		xvec(1) = R_P - dir*r*0.05d0*cos(theta)
		xvec(2) = y_P - dir*r*0.05d0*sin(theta)

		call newt(ntrial,xvec,2,newton_check,tolx,tolf)

		dist = ( (xvec(1) - R_P)**2 + (xvec(2) - y_P)**2 )**0.5d0

	endif

	angle = atan2((y_P - xvec(2)),(R_P - xvec(1)))



end subroutine get_dist


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine get_dist_old(iloc,jloc,dist,dir, angle)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! calculate the distance from the boundary

	use mod_parameters, only : rmajor, xlength, ylength, pi
	use mod_arrays, only : xV, yV

	real(kind=dkind) :: dist, dir
	integer :: iloc, jloc
	real(kind=dkind) :: angle
	real(kind=dkind) :: ex, ey, r, rprim, rsec, theta
	real(kind=dkind) :: xvec(2)
	integer :: ntrial = 50000
	real(kind=dkind) :: tolx=1.d-12
	real(kind=dkind) :: tolf = 1.d-9
	logical :: newton_check
	real(kind=dkind), dimension(1:3) :: theta_start
	real(kind=dkind) :: thetamin

	R_P = xV(iloc) + rmajor - xlength/2.d0
	y_P = yV(jloc) - ylength/2.d0

!	call radius(iloc,jloc,ex,ey,r)
	call radius_der(R_P,y_P,ex,ey,theta,r,rprim,rsec)

	if(ex**2+ey**2>r**2) then
		dir = 1.d0	! external point
	else
		dir = -1.d0	! internal point
	endif

!	! patch for obscure bug
!	if((abs(xV(iloc)-xlength/2.d0)<xlength/10.d0).and.(abs(yV(jloc)-ylength/2.d0)<ylength/10.d0)) then
!	! the point is close to the center
!
!		dist = 1.d0
!		dir = -1.d0
!		return
!
!	endif

	! patch for obscure bug: use minimization near the plasma center
	if((abs(xV(iloc)-xlength/2.d0)<xlength/3.d0).and.(abs(yV(jloc)-ylength/2.d0)<ylength/3.d0)) then
	! the point is close to the center

		if(yV(jloc)-ylength/2.d0>=0.d0) then
		! upper region

			if(xV(iloc)-xlength/2.d0>0d0) then
			! top-right quadrant

				theta_start(1) = 0.d0
				theta_start(2) = pi/4.d0
				theta_start(3) = 1.2d0*pi/2.d0

			else
			! top-left quadrant

				theta_start(1) = 0.8d0*pi/2.d0
				theta_start(2) = pi*0.75d0
				theta_start(3) = pi

			endif

		else

			if(xV(iloc)-xlength/2.d0>0d0) then
			! bottom-right quadrant

				theta_start(1) = 1.4d0*pi
				theta_start(2) = pi*1.75d0
				theta_start(3) = 2.d0*pi

			else
			! bottom-left quadrant

				theta_start(1) = pi
				theta_start(2) = pi*1.25d0
				theta_start(3) = pi*1.6d0

			endif

!			theta_start(1) = pi
!			theta_start(2) = pi*1.5d0
!			theta_start(3) = 2.d0*pi

		endif

		dist = brent(theta_start(1), theta_start(2), theta_start(3), dist_fun, tolf, thetamin)

		call radius_theta(thetamin,r,xvec(1),xvec(2))

	else

		xvec(1) = R_P - dir*r*0.05d0*cos(theta)
		xvec(2) = y_P - dir*r*0.05d0*sin(theta)

		call newt(ntrial,xvec,2,newton_check,tolx,tolf)

		dist = ( (xvec(1) - R_P)**2 + (xvec(2) - y_P)**2 )**0.5d0

	endif

	angle = atan2((y_P - xvec(2)),(R_P - xvec(1)))



end subroutine get_dist_old


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine resistivity_vals(dist,res,dres)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! res is the resistivity value, dres the derivative with respect to r

	real(kind=dkind) :: dist, res, dres
	real(kind=dkind) :: jump_dist

	jump_dist = dist - res_eta_xjump

	res = res_eta_min + (res_eta_max-res_eta_min) * (1.d0 + tanh(res_factor*jump_dist))/2.d0

	dres = res_factor*(res_eta_max-res_eta_min)/2.d0/cosh(res_factor*jump_dist)**2

end subroutine resistivity_vals

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine newt(maxits,x,n,check,tolx,tolf)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@


	implicit none

	INTEGER n,nn,NP,MAXITS
	LOGICAL check
	REAL(kind=dkind) :: x(n),fvec,TOLF,TOLMIN,TOLX,STPMX
	PARAMETER  (NP=40,TOLMIN=1.d-6,STPMX=100.d0)
	COMMON /newtv/ fvec(NP),nn
	SAVE /newtv/
!	CU    USES fdjac,fmin,lnsrch,lubksb,ludcmp
	INTEGER i,its,j,indx(NP)
	REAL(kind=dkind) d,den,f,fold,stpmax,sum,temp,test,fjac(NP,NP),g(NP),  &
					p(NP),xold(NP) !,fmin
!	EXTERNAL fmin

	nn=n
	f=fmin(x)
	test=0.d0

	do 11 i=1,n
	if(abs(fvec(i)).gt.test)test=abs(fvec(i))
11    continue
	if(test.lt..01d0*TOLF)return
	sum=0.d0
	do 12 i=1,n
	sum=sum+x(i)**2
12    continue
	stpmax=STPMX*max(sqrt(sum),real(n,dkind))
	do 21 its=1,MAXITS
	call fdjac(n,x,fvec,NP,fjac)
	do 14 i=1,n
	  sum=0.d0
	  do 13 j=1,n
		sum=sum+fjac(j,i)*fvec(j)
13        continue
	  g(i)=sum
14      continue
	do 15 i=1,n
	  xold(i)=x(i)
15      continue
	fold=f
	do 16 i=1,n
	  p(i)=-fvec(i)
16      continue
	call ludcmp(fjac,n,NP,indx,d)
	call lubksb(fjac,n,NP,indx,p)
	call lnsrch(n,xold,fold,g,p,x,f,stpmax,check,fmin)
	test=0.d0
	do 17 i=1,n
	  if(abs(fvec(i)).gt.test)test=abs(fvec(i))
17      continue
	if(test.lt.TOLF)then
	  check=.false.
	  return
	endif
	if(check)then
	  test=0.d0
	  den=max(f,.5d0*n)
	  do 18 i=1,n
		temp=abs(g(i))*max(abs(x(i)),1.d0)/den
		if(temp.gt.test)test=temp
18        continue
	  if(test.lt.TOLMIN)then
		check=.true.
	  else
		check=.false.
	  endif
	  return
	endif
	test=0.d0
	do 19 i=1,n
	  temp=(abs(x(i)-xold(i)))/max(abs(x(i)),1.d0)
	  if(temp.gt.test)test=temp
	19      continue
	if(test.lt.TOLX)return
	21    continue

	print*, 'MAXITS exceeded in newt'

end subroutine newt


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine fdjac(n,x,f,m,jac)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! only computes the Jacobian, f is not needed,
! but passed for library compatibility

	use mod_parameters, only : rmajor

	implicit none

	integer :: n,m	! just to keep compatibility with the library, n=m
	real(kind=dkind), dimension(1:m) :: x,f
	real(kind=dkind), dimension(1:m,1:m) :: jac
	integer :: zone
	real(kind=dkind) :: theta, ex, ey, r, rprim, rsec, rloc, rloc2, rloc3, a_x, a_y


	call radius_der(x(1),x(2),ex,ey,theta,r,rprim,rsec)
	! NOTE: ey=x(2)!

	rloc2 = ex**2+ey**2
	rloc = sqrt(rloc2)
	rloc3 = rloc**3
	a_x = ex*(x(1)-R_P)
	a_y = ey*(x(2)-y_P)

!	jac(1,1) = - y_P +  &
!					( ex**3 + ex*ey*y_P + (x(1)-R_P)*ey**2 ) * rprim/rloc3 +  &
!					( -ex*ey*(x(1)-R_P) - ey**2*(x(2)-y_P) ) * rsec/rloc3

!	jac(1,2) = -(R_P-rmajor) +  &
!					( ex**2*ey - ex*ey*(rmajor-R_P) + ey**3 - ex**2*x(2) ) * rprim/rloc3 +  &
!					( ex**2*(x(1)-R_P) - ex*ey*(y_P-x(2)) ) * rsec/rloc3


	jac(1,1) = -y_P +  &
					( (ex+x(1)-R_P)/rloc - ex*(a_x+a_y)/rloc3) * rprim +  &
					ey*(a_x-a_y)/rloc3 * rsec

	jac(1,2) = R_P-rmajor +  &
					( (2.d0*x(2)-y_P)/rloc - ey*(a_x+a_y)/rloc3) * rprim +  &
					ex*(a_x+a_y)/rloc3 * rsec

	jac(2,1) = 2.d0 * ( ex + ey/rloc2*r*rprim )

	jac(2,2) = 2.d0 * ( ey - ex/rloc2*r*rprim )


	continue

end subroutine fdjac


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
function fmin(x)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	implicit none

	INTEGER n,NP
	REAL(kind=dkind) ::  fmin,x(*),fvec
	PARAMETER (NP=40)
	COMMON /newtv/ fvec(NP),n
	SAVE /newtv/

!	CU    USES funcv
	INTEGER i
	REAL(kind=dkind) :: sum
	call funcv(n,x,fvec)
	sum=0.d0
	do 11 i=1,n
	sum=sum+fvec(i)**2
	11    continue
	fmin=0.5d0*sum
	return

end function fmin


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine funcv(n,x,f)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
!this only computes the value of the function, to be used by newt

	use mod_parameters, only : rmajor

	implicit none

	integer :: n
	real(kind=dkind), dimension(1:n) :: x,f
	real(kind=dkind) :: theta, ex, ey, r, rprim, rsec, rloc, rloc2


	call radius_der(x(1),x(2),ex,ey,theta,r,rprim,rsec)
	! NOTE: ey=x(2)!

	rloc2 = ex**2+ey**2
	rloc = sqrt(rloc2)

	f(1) = ( ex*rprim/rloc - ey ) * (x(1)-R_P) +  &
			( ey*rprim/rloc + ex ) * (x(2)-y_P)

	f(2) = ex**2 + ey**2 - r**2


	continue

end subroutine funcv

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
function dist_fun(theta) result(dist)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	real(kind=dkind) :: theta, dist
	real(kind=dkind) :: r, Rloc, yloc

	call radius_theta(theta,r,Rloc,yloc)

	dist = sqrt((Rloc-R_P)**2+(yloc-y_P)**2)

end function dist_fun

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
function brent(ax,bx,cx,f,tol,xmin)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	! calculates the minimum of the input function f
	! the minimum point is returned as xmin, the minimum value as output of the function

	implicit none

	integer, parameter :: itmax=100
	real(dkind), parameter :: cgold=0.3819660112501051d0
	real(dkind), parameter :: zeps=1.0d-10
	real(dkind) :: brent

	real(dkind) :: a, b, ax, bx, cx, tol, xmin
	real(dkind) :: v, w, x, e, fx, fv, fw, xm, tol1, tol2, r, q, p, etemp, d, u, fu

	real(dkind) :: f

	integer :: iter

	external f

      a=min(ax,cx)
      b=max(ax,cx)
      v=bx
      w=v
      x=v
      e=0.
      fx=f(x)
      fv=fx
      fw=fx
      do 11 iter=1,itmax
        xm=0.5*(a+b)
        tol1=tol*abs(x)+zeps
        tol2=2.*tol1
        if(abs(x-xm).le.(tol2-.5*(b-a))) goto 3
        if(abs(e).gt.tol1) then
          r=(x-w)*(fx-fv)
          q=(x-v)*(fx-fw)
          p=(x-v)*q-(x-w)*r
          q=2.*(q-r)
          if(q.gt.0.) p=-p
          q=abs(q)
          etemp=e
          e=d
          if(abs(p).ge.abs(.5*q*etemp).or.p.le.q*(a-x).or.p.ge.q*(b-x)) goto 1
          d=p/q
          u=x+d
          if(u-a.lt.tol2 .or. b-u.lt.tol2) d=sign(tol1,xm-x)
          goto 2
        endif
1       if(x.ge.xm) then
          e=a-x
        else
          e=b-x
        endif
        d=cgold*e
2       if(abs(d).ge.tol1) then
          u=x+d
        else
          u=x+sign(tol1,d)
        endif
        fu=f(u)
        if(fu.le.fx) then
          if(u.ge.x) then
            a=x
          else
            b=x
          endif
          v=w
          fv=fw
          w=x
          fw=fx
          x=u
          fx=fu
        else
          if(u.lt.x) then
            a=u
          else
            b=u
          endif
          if(fu.le.fw .or. w.eq.x) then
            v=w
            fv=fw
            w=u
            fw=fu
          else if(fu.le.fv .or. v.eq.x .or. v.eq.w) then
            v=u
            fv=fu
          endif
        endif
11    continue
      pause 'brent exceed maximum iterations.'
3     xmin=x
      brent=fx

      return

end function brent

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine resistivity_physical
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	use mod_parameters, only : nx, ny, dx, dy, rho_boundary, p_boundary, restart_option
	use mod_arrays, only : p, rho

	real(kind=dkind), save :: res_0 = 0.d0

	integer :: i, j

	if((res_implicit_option==45).and.(initialize_eta_phys)) then
		call setup_eta_phys
		initialize_eta_phys = .false.
		restart_option = 2
	endif

	if(res_0==0.d0) then

		if((res_implicit_option==4).or.(res_implicit_option==14)) then
			res_0 = res_eta_max * (p_boundary/rho_boundary)**res_phys_exp
		elseif((res_implicit_option==5).or.(res_implicit_option==15)) then
			res_0 = res_eta_max * rho_boundary**res_phys_exp
		elseif((res_implicit_option==6).or.(res_implicit_option==16)) then
			res_0 = res_eta_max * p_boundary**res_phys_exp
		elseif(res_implicit_option==45) then
			res_0 = res_eta_max * eta_phys_boundary**res_phys_exp
		endif

	endif

	if(res_implicit_option==4) then

		!$omp parallel default(shared)
		!$omp do private(i,j)
		do j = 0, ny+1
		do i = 0, nx+1

			res_eta(i,j,0) = res_0 * (p(i,j)/rho(i,j))**(-res_phys_exp)

		enddo
		enddo
		!$omp enddo
		!$omp end parallel

	elseif(res_implicit_option==5) then

		!$omp parallel default(shared)
		!$omp do private(i,j)
		do j = 0, ny+1
		do i = 0, nx+1

			res_eta(i,j,0) = res_0 * rho(i,j)**(-res_phys_exp)

		enddo
		enddo
		!$omp enddo
		!$omp end parallel

	elseif(res_implicit_option==6) then

		!$omp parallel default(shared)
		!$omp do private(i,j)
		do j = 0, ny+1
		do i = 0, nx+1

			res_eta(i,j,0) = res_0 * p(i,j)**(-res_phys_exp)

		enddo
		enddo
		!$omp enddo
		!$omp end parallel

	elseif(res_implicit_option==14) then

		!$omp parallel default(shared)
		!$omp do private(i,j)
		do j = 0, ny+1
		do i = 0, nx+1

			res_eta(i,j,0) = min( res_0 * (p(i,j)/rho(i,j))**(-res_phys_exp), res_eta_max)

		enddo
		enddo
		!$omp enddo
		!$omp end parallel

	elseif(res_implicit_option==15) then

		!$omp parallel default(shared)
		!$omp do private(i,j)
		do j = 0, ny+1
		do i = 0, nx+1

			res_eta(i,j,0) = min( res_0 * rho(i,j)**(-res_phys_exp), res_eta_max)

		enddo
		enddo
		!$omp enddo
		!$omp end parallel

	elseif(res_implicit_option==16) then

		!$omp parallel default(shared)
		!$omp do private(i,j)
		do j = 0, ny+1
		do i = 0, nx+1

			res_eta(i,j,0) = min( res_0 * p(i,j)**(-res_phys_exp), res_eta_max)

		enddo
		enddo
		!$omp enddo
		!$omp end parallel

	elseif(res_implicit_option==25) then

		!$omp parallel default(shared)
		!$omp do private(i,j)
		do j = 0, ny+1
		do i = 0, nx+1

			res_eta(i,j,0) = res_eta_max * resistivity_numerical(rho(i,j))

		enddo
		enddo
		!$omp enddo
		!$omp end parallel

	elseif(res_implicit_option==35) then

		!$omp parallel default(shared)
		!$omp do private(i,j)
		do j = 0, ny+1
		do i = 0, nx+1

			res_eta(i,j,0) = res_eta_max * resistivity_analytical(rho(i,j))

		enddo
		enddo
		!$omp enddo
		!$omp end parallel

	elseif(res_implicit_option==45) then

		!$omp parallel default(shared)
		!$omp do private(i,j)
		do j = 0, ny+1
		do i = 0, nx+1

			res_eta(i,j,0) = min( res_0 * eta_phys(i,j)**(-res_phys_exp), res_eta_max )

		enddo
		enddo
		!$omp enddo
		!$omp end parallel

	endif

	do j = 1, ny
	do i = 1, nx

		res_eta(i,j,1) = (res_eta(i+1,j,0)-res_eta(i-1,j,0))/(2.d0*dx)
		res_eta(i,j,2) = (res_eta(i,j+1,0)-res_eta(i,j-1,0))/(2.d0*dy)
!		res_eta(i,j,1:2) = 0.d0

	enddo
	enddo

	continue

	return

end subroutine resistivity_physical

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine setup_eta_phys
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	use mod_parameters, only : nx, ny

	real(kind=dkind) :: jump_dist
	integer :: i, j

	eta_phys = eta_phys_boundary

	do j = 0, ny+1
	do i = 0, nx+1

		jump_dist = edge_distance(i,j) - res_eta_xjump

		eta_phys(i,j) = eta_phys_center - (eta_phys_center-eta_phys_boundary) * (1.d0 + tanh(res_factor*jump_dist))/2.d0

		continue

	enddo
	enddo

	continue

	return

end subroutine setup_eta_phys

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
function resistivity_numerical(x) result(answer)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	use pseudo_IMSL, only : dbsval

	real(kind=dkind) :: x, answer

	if(x<=res_data(1,1)) then
		answer = res_data(1,2)
	elseif(x>=res_data(resistivity_points,1)) then
		answer = res_data(resistivity_points,2)
	else
		answer =  dbsval(x, res_ord, res_data(:,3),  &
						resistivity_points, res_cscoef(1,1:resistivity_points) )
	endif

	continue

end function resistivity_numerical

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
function resistivity_analytical(x) result(answer)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	real(kind=dkind) :: x, answer

	if(res_scale==0.d0) res_scale =  (1.d0-tanh(res_eta_xjump*(-0.5d0*res_xmin)/res_xmin))/2.d0

	if(x<=res_xmin) then
		answer = 1.d0
	elseif(x>=res_xmax) then
		answer = res_eta_min
	else
		answer =  res_eta_min + (1.d0-res_eta_min) * (1.d0-tanh(res_eta_xjump*(x-1.5d0*res_xmin)/res_xmin))/2.d0 / res_scale
	endif

	continue

end function resistivity_analytical

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine resistive_diffusion(icycle)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	integer :: icycle

	if(res_eta_max==0.d0) return

	if(res_implicit_option==-1) then
		call resistive_diffusion_0old(icycle)
	elseif(res_implicit_option==0) then
		call resistive_diffusion_0(icycle)
	elseif(res_implicit_option==1) then
		call resistive_diffusion_1(icycle)
	elseif(res_implicit_option==2) then
		call resistive_diffusion_2(icycle)
	elseif(res_implicit_option==3) then
		call resistive_diffusion_3(icycle)
	elseif((res_implicit_option==4).or.(res_implicit_option==5).or.(res_implicit_option==6).or.  &
			(res_implicit_option==14).or.(res_implicit_option==15).or.(res_implicit_option==16)  &
			.or.(res_implicit_option==25).or.(res_implicit_option==35).or.(res_implicit_option==45)) then
	! first update the resistivity as function of  rho, T, or p, then call the diffusion
		call resistivity_physical
		call resistive_diffusion_2(icycle)
	endif

end subroutine resistive_diffusion

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine resistive_diffusion_0old(icycle)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! ADI diffusion for magnetic field

	use mod_parameters, only : nx, ny, dt, dx, dy
	use mod_arrays, only : nmx, nmy, bx, by, bz

	integer :: icycle
	integer :: direction(2)
	real(kind=dkind), dimension(:,:), pointer :: Bfield
	real(kind=dkind), dimension(:), allocatable, save :: a, b, c, r, Bnew
	real(kind=dkind) :: dtdx2, dtdy2
	integer :: i, j, k, istep
	integer :: i_p, j_p ! indexes for pointer

	if(allocated(a)) then
		continue
	else
		allocate(a(0:max(nx,ny)+1), b(0:max(nx,ny)+1),  &
					c(0:max(nx,ny)+1), r(0:max(nx,ny)+1),  &
					Bnew(0:max(nx,ny)+1) )
	endif

	nullify(Bfield)

	! half time steps
	dtdx2 = dt/dx**2 / 2.d0
	dtdy2 = dt/dy**2 / 2.d0

	if(modulo(icycle,2)==0) then

		direction(1) = 1
		direction(2) = 2

	elseif(modulo(icycle,2)==1) then

		direction(1) = 2
		direction(2) = 1

	endif

	do k = 1, 3
	! "k" is the field component

		if(k==1) then
			Bfield => bx(0:nx+1,0:ny+1)
		elseif(k==2) then
			Bfield => by(0:nx+1,0:ny+1)
		elseif(k==3) then
			Bfield => bz(0:nx+1,0:ny+1)
		endif

		do istep = 1, 2

			if(direction(istep)==1) then
			! do diffusion in the x direction (y is treated explicitly)

				do j = 1, ny

					j_p = j+1
					! pointer index is shifted by 1!

					! first build the coefficients

					do i = 1, nx

						a(i) = -dtdx2 * res_eta(i,j,0)
						b(i) = 1.d0 + 2.d0*dtdx2 * res_eta(i,j,0)
						c(i) = -dtdx2 * res_eta(i,j,0)

					enddo

					! apply fixed boundary conditions

					a(0) = 0.d0; c(0) = 0.d0
					b(0) = 1.d0
					a(nx+1) = 0.d0; c(nx+1) = 0.d0
					b(nx+1) = 1.d0

					! build the RHS

					do i = 1, nx

						i_p = i+1
						! pointer index is shifted by 1!

						r(i) = Bfield(i_p,j_p) +  &
								res_eta(i,j,0) * dtdy2 *  &
								( Bfield(i_p,j_p+1) - 2.d0*Bfield(i_p,j_p) + Bfield(i_p,j_p-1) )

					enddo

					r(0) = Bfield(1,j_p)
					r(nx+1) = Bfield(nx+2,j_p)

					! solve the tridiagonal system

					call tridiag(a,b,c,r,Bnew,nx+2,nx+2)

					! update the field

					do i = 1, nx

						i_p = i+1
						! pointer index is shifted by 1!

						Bfield(i_p,j_p) = Bnew(i)

					enddo

				enddo

			elseif(direction(istep)==2) then
			! do diffusion in the y direction (x is treated explicitly)

				do i = 1, nx

					i_p = i+1
					! pointer index is shifted by 1!

					! first build the coefficients

					do j = 1, ny

						a(j) = -dtdy2 * res_eta(i,j,0)
						b(j) = 1.d0 + 2.d0*dtdy2 * res_eta(i,j,0)
						c(j) = -dtdy2 * res_eta(i,j,0)

					enddo

					! apply fixed boundary conditions

					a(0) = 0.d0; c(0) = 0.d0
					b(0) = 1.d0
					a(ny+1) = 0.d0; c(ny+1) = 0.d0
					b(ny+1) = 1.d0

					! build the RHS

					do j = 1, ny

						j_p = j+1
						! pointer index is shifted by 1!

						r(j) = Bfield(i_p,j_p) +  &
								res_eta(i,j,0) * dtdx2 *  &
								( Bfield(i_p+1,j_p) - 2.d0*Bfield(i_p,j_p) + Bfield(i_p-1,j_p) )

					enddo

					r(0) = Bfield(i_p,1)
					r(ny+1) = Bfield(i_p,ny+2)

					! solve the tridiagonal system

					call tridiag(a,b,c,r,Bnew,ny+2,ny+2)

					! update the field

					do j = 1, ny

						j_p = j+1
						! pointer index is shifted by 1!

						Bfield(i_p,j_p) = Bnew(j)

					enddo

				enddo

			endif

		enddo

		nullify(Bfield)

	enddo

end subroutine resistive_diffusion_0old

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine resistive_diffusion_0(icycle)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! ADI diffusion for magnetic field

	use mod_parameters, only : nx, ny, dt, dx, dy
	use mod_arrays, only : nmx, nmy, bx, by, bz

	integer :: icycle
	integer :: direction(2)
	real(kind=dkind), dimension(:,:), pointer :: Bfield
	real(kind=dkind), dimension(:), allocatable, save :: a, b, c, r
	real(kind=dkind), dimension(:,:), allocatable, save :: Bnew
	real(kind=dkind) :: dtdx2, dtdy2
	integer :: i, j, k, istep
	integer :: i_p, j_p ! indexes for pointer

	if(allocated(a)) then
		continue
	else
		allocate(a(0:max(nx,ny)+1), b(0:max(nx,ny)+1),  &
					c(0:max(nx,ny)+1), r(0:max(nx,ny)+1),  &
					Bnew(0:nx+1,0:ny+1) )
	endif

	nullify(Bfield)

	! half time steps
	dtdx2 = dt/dx**2 / 2.d0
	dtdy2 = dt/dy**2 / 2.d0

	if(modulo(icycle,2)==0) then

		direction(1) = 1
		direction(2) = 2

	elseif(modulo(icycle,2)==1) then

		direction(1) = 2
		direction(2) = 1

	endif

	do k = 1, 3
	! "k" is the field component

		if(k==1) then
			Bfield => bx(0:nx+1,0:ny+1)
		elseif(k==2) then
			Bfield => by(0:nx+1,0:ny+1)
		elseif(k==3) then
			Bfield => bz(0:nx+1,0:ny+1)
		endif

		do istep = 1, 2

			if(direction(istep)==1) then
			! do diffusion in the x direction (y is treated explicitly)

				do j = 1, ny

					j_p = j+1
					! pointer index is shifted by 1!

					! first build the coefficients

					do i = 1, nx

						a(i) = -dtdx2 * res_eta(i,j,0)
						b(i) = 1.d0 + 2.d0*dtdx2 * res_eta(i,j,0)
						c(i) = -dtdx2 * res_eta(i,j,0)

					enddo

					! apply fixed boundary conditions

					a(0) = 0.d0; c(0) = 0.d0
					b(0) = 1.d0
					a(nx+1) = 0.d0; c(nx+1) = 0.d0
					b(nx+1) = 1.d0

					! build the RHS

					do i = 1, nx

						i_p = i+1
						! pointer index is shifted by 1!

						r(i) = Bfield(i_p,j_p) +  &
								res_eta(i,j,0) * dtdy2 *  &
								( Bfield(i_p,j_p+1) - 2.d0*Bfield(i_p,j_p) + Bfield(i_p,j_p-1) )

					enddo

					r(0) = Bfield(1,j_p)
					r(nx+1) = Bfield(nx+2,j_p)

					! solve the tridiagonal system

					call tridiag(a,b,c,r,Bnew(:,j),nx+2,nx+2)

				enddo

			elseif(direction(istep)==2) then
			! do diffusion in the y direction (x is treated explicitly)

				do i = 1, nx

					i_p = i+1
					! pointer index is shifted by 1!

					! first build the coefficients

					do j = 1, ny

						a(j) = -dtdy2 * res_eta(i,j,0)
						b(j) = 1.d0 + 2.d0*dtdy2 * res_eta(i,j,0)
						c(j) = -dtdy2 * res_eta(i,j,0)

					enddo

					! apply fixed boundary conditions

					a(0) = 0.d0; c(0) = 0.d0
					b(0) = 1.d0
					a(ny+1) = 0.d0; c(ny+1) = 0.d0
					b(ny+1) = 1.d0

					! build the RHS

					do j = 1, ny

						j_p = j+1
						! pointer index is shifted by 1!

						r(j) = Bfield(i_p,j_p) +  &
								res_eta(i,j,0) * dtdx2 *  &
								( Bfield(i_p+1,j_p) - 2.d0*Bfield(i_p,j_p) + Bfield(i_p-1,j_p) )

					enddo

					r(0) = Bfield(i_p,1)
					r(ny+1) = Bfield(i_p,ny+2)

					! solve the tridiagonal system

					call tridiag(a,b,c,r,Bnew(i,:),ny+2,ny+2)

				enddo

			endif

			! update the field

			do j = 1, ny

				j_p = j+1 
				! pointer index is shifted by 1!

				do i = 1, nx

					i_p = i+1
					! pointer index is shifted by 1!

					Bfield(i_p,j_p) = Bnew(i,j)

				enddo

			enddo

		enddo

		nullify(Bfield)

	enddo

end subroutine resistive_diffusion_0

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine resistive_diffusion_1(icycle)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! ADI diffusion for magnetic field

	use mod_parameters, only : nx, ny, dt, dx, dy
	use mod_arrays, only : nmx, nmy, bx, by, bz, Rmaj

	integer :: icycle
	integer :: direction(2)
	real(kind=dkind), dimension(:,:), pointer :: Bfield
	real(kind=dkind), dimension(:), allocatable, save :: a, b, c, r
	real(kind=dkind), dimension(:,:), allocatable, save :: Bnew
	real(kind=dkind) :: dtdx2, dtdy2
	integer :: i, j, k, istep
	integer :: i_p, j_p ! indexes for pointer

	if(allocated(a)) then
		continue
	else
		allocate(a(0:max(nx,ny)+1), b(0:max(nx,ny)+1),  &
					c(0:max(nx,ny)+1), r(0:max(nx,ny)+1),  &
					Bnew(0:nx+1,0:ny+1) )
	endif

	nullify(Bfield)

	! half time steps
	dtdx2 = dt/dx**2 / 2.d0
	dtdy2 = dt/dy**2 / 2.d0

	if(modulo(icycle,2)==0) then

		direction(1) = 1
		direction(2) = 2

	elseif(modulo(icycle,2)==1) then

		direction(1) = 2
		direction(2) = 1

	endif

	do k = 1, 3
	! "k" is the field component

		if(k==1) then
			Bfield => bx(0:nx+1,0:ny+1)
		elseif(k==2) then
			Bfield => by(0:nx+1,0:ny+1)
		elseif(k==3) then
			Bfield => bz(0:nx+1,0:ny+1)
		endif

		do istep = 1, 2

			if(direction(istep)==1) then
			! do diffusion in the x direction (y is treated explicitly)

				do j = 1, ny

					j_p = j+1
					! pointer index is shifted by 1!

					! first build the coefficients

					do i = 1, nx

						a(i) = -res_eta(i,j,0) * dtdx2 * ( 1.d0 - dx/(2.d0*Rmaj(i)) )
						b(i) = 1.d0 + 2.d0*dtdx2 * res_eta(i,j,0)
						c(i) = -res_eta(i,j,0) * dtdx2 * ( 1.d0 + dx/(2.d0*Rmaj(i)) )

						if(k/=2) then
							b(i) = b(i) + res_eta(i,j,0) * dt/(2.d0*Rmaj(i)**2)
						endif

					enddo

					! apply fixed boundary conditions

					a(0) = 0.d0; c(0) = 0.d0
					b(0) = 1.d0
					a(nx+1) = 0.d0; c(nx+1) = 0.d0
					b(nx+1) = 1.d0

					! build the RHS

					do i = 1, nx

						i_p = i+1
						! pointer index is shifted by 1!

						r(i) = Bfield(i_p,j_p) +  &
								res_eta(i,j,0) *  &
								( dtdy2 * ( Bfield(i_p,j_p+1) - 2.d0*Bfield(i_p,j_p) + Bfield(i_p,j_p-1) ) )

					enddo

					r(0) = Bfield(1,j_p)
					r(nx+1) = Bfield(nx+2,j_p)

					! solve the tridiagonal system

					call tridiag(a,b,c,r,Bnew(:,j),nx+2,nx+2)

				enddo

			elseif(direction(istep)==2) then
			! do diffusion in the y direction (x is treated explicitly)

				do i = 1, nx

					i_p = i+1
					! pointer index is shifted by 1!

					! first build the coefficients

					do j = 1, ny

						a(j) = -dtdy2 * res_eta(i,j,0)
						b(j) = 1.d0 + 2.d0*dtdy2 * res_eta(i,j,0)
						c(j) = -dtdy2 * res_eta(i,j,0)

					enddo

					! apply fixed boundary conditions

					a(0) = 0.d0; c(0) = 0.d0
					b(0) = 1.d0
					a(ny+1) = 0.d0; c(ny+1) = 0.d0
					b(ny+1) = 1.d0

					! build the RHS

					do j = 1, ny

						j_p = j+1
						! pointer index is shifted by 1!

						r(j) = Bfield(i_p,j_p) +  &
								res_eta(i,j,0) *  &
								( dtdx2 * ( Bfield(i_p+1,j_p) - 2.d0*Bfield(i_p,j_p) + Bfield(i_p-1,j_p) )  &
								  + dt/(4.d0*dx*Rmaj(i)) * ( Bfield(i_p+1,j_p) -Bfield(i_p-1,j_p) )  )

						if(k/=2) then
							r(i) = r(i) - res_eta(i,j,0) * dt/(2.d0*Rmaj(i)**2)*Bfield(i_p,j_p)
						endif

					enddo

					r(0) = Bfield(i_p,1)
					r(ny+1) = Bfield(i_p,ny+2)

					! solve the tridiagonal system

					call tridiag(a,b,c,r,Bnew(i,:),ny+2,ny+2)

				enddo

			endif

			! update the field

			do j = 1, ny

				j_p = j+1 
				! pointer index is shifted by 1!

				do i = 1, nx

					i_p = i+1
					! pointer index is shifted by 1!

					Bfield(i_p,j_p) = Bnew(i,j)

				enddo

			enddo

		enddo

		nullify(Bfield)

	enddo

end subroutine resistive_diffusion_1

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine resistive_diffusion_2(icycle)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! ADI diffusion for magnetic field
! for the sake of clearness, equations are repeated for each component

	use mod_parameters, only : nx, ny, dt, dx, dy
	use mod_arrays, only : nmx, nmy, bx, by, bz, Rmaj

	integer :: icycle
	integer :: direction(2)
	real(kind=dkind), dimension(:,:), pointer :: Bfield
	real(kind=dkind), dimension(:), allocatable, save :: a, b, c, r
	real(kind=dkind), dimension(:,:), allocatable, save :: Bnew
	real(kind=dkind), dimension(:), allocatable, save :: Btemp_i, Btemp_j
	real(kind=dkind) :: dtdx2, dtdy2
	integer :: i, j, k, istep, kwhich, i_additional
        integer :: icycle_temp
	integer :: i_p, j_p ! indexes for pointer
	integer, dimension(1:3), save :: which_component = 0

	if(allocated(a)) then
		continue
	else
		allocate(a(0:max(nx,ny)+1), b(0:max(nx,ny)+1),  &
					c(0:max(nx,ny)+1), r(0:max(nx,ny)+1),  &
					Bnew(0:nx+1,0:ny+1), Btemp_i(0:nx+1),Btemp_j(0:ny+1) )
	endif

	if(maxval(which_component)==0) then
	! set up initial values

		which_component(1) = 1
		which_component(2) = 2
		which_component(3) = 3

	endif

	nullify(Bfield)

        icycle_temp = icycle
        dt = dt / (additional_res_steps+1.d0)

	! half time steps
	dtdx2 = dt/dx**2 / 2.d0
	dtdy2 = dt/dy**2 / 2.d0

        ! cycle on the smaller local time step

        do i_additional = 1, additional_res_steps+1

	if(modulo(icycle_temp,2)==0) then

		direction(1) = 1
		direction(2) = 2

	elseif(modulo(icycle_temp,2)==1) then

		direction(1) = 2
		direction(2) = 1

	endif

	do istep = 1, 2

		do k = 1, 3
		! "k" is the field component

			kwhich = which_component(k)

			if(kwhich==1) then
				Bfield => bx(0:nx+1,0:ny+1)
			elseif(kwhich==2) then
				Bfield => by(0:nx+1,0:ny+1)
			elseif(kwhich==3) then
				Bfield => bz(0:nx+1,0:ny+1)
			endif

!			if(kwhich/=3) cycle

			if(direction(istep)==1) then
			! do diffusion in the x direction (y is treated explicitly)

!!$			!$omp parallel default(shared)
!!$			!$omp do private(i,j,a,b,c,r,i_p,j_p)

				do j = 1, ny

					j_p = j+1
					! pointer index is shifted by 1!

					! first build the coefficients

!!$					!$omp do private(i)
					do i = 1, nx

						if(kwhich==1) then
						! Bx equation

							a(i) = -res_eta(i,j,0) * dtdx2 * ( 1.d0 - dx/(2.d0*Rmaj(i)) )

							b(i) = 1.d0 + res_eta(i,j,0) * (2.d0*dtdx2 +  dt/(2.d0*Rmaj(i)**2) )

							c(i) = -res_eta(i,j,0) * dtdx2 * ( 1.d0 + dx/(2.d0*Rmaj(i)) )

						elseif(kwhich==2) then
						! By equation

							a(i) = -res_eta(i,j,0) * dtdx2 * ( 1.d0 - dx/(2.d0*Rmaj(i)) )  &
										+res_eta(i,j,1) * dt/(4.d0*dx)

							b(i) = 1.d0 + res_eta(i,j,0) * 2.d0*dtdx2

							c(i) = -res_eta(i,j,0) * dtdx2 * ( 1.d0 + dx/(2.d0*Rmaj(i)) )  &
										-res_eta(i,j,1) * dt/(4.d0*dx)

						elseif(kwhich==3) then
						! Bz equation

							a(i) = -res_eta(i,j,0) * dtdx2 * ( 1.d0 - dx/(2.d0*Rmaj(i)) )  &
										-res_eta(i,j,1) * dt/(4.d0*dx)

							b(i) = 1.d0 + res_eta(i,j,0) * (2.d0*dtdx2 +  dt/(2.d0*Rmaj(i)**2) )  &
												+res_eta(i,j,1) * dt/(2.d0*Rmaj(i))
	!						b(i) = 1.d0 + res_eta(i,j,0) * (2.d0*dtdx2 +  dt/(2.d0*Rmaj(i)**2) )  &
	!											+res_eta(i,j,1) * dt/(2.d0*Rmaj(i))

							c(i) = -res_eta(i,j,0) * dtdx2 * ( 1.d0 + dx/(2.d0*Rmaj(i)) )  &
										+res_eta(i,j,1) * dt/(4.d0*dx)

						endif

					enddo
!!$					!$omp enddo

					! apply fixed boundary conditions

					a(0) = 0.d0; c(0) = 0.d0
					b(0) = 1.d0
					a(nx+1) = 0.d0; c(nx+1) = 0.d0
					b(nx+1) = 1.d0

					! build the RHS
					! for the sake of clearness, use explicit definitions

!!$					!$omp do private(i)
					do i = 1, nx

						if(kwhich==1) then

							r(i) = bx(i,j) + res_eta(i,j,0) *dtdy2 * (bx(i,j+1)-2.d0*bx(i,j)+bx(i,j-1)) +  &
												dt/4.d0 * res_eta(i,j,2) * ( (bx(i,j+1)-bx(i,j-1))/dy - (by(i+1,j)-by(i-1,j))/dx )

						elseif(kwhich==2) then

							r(i) = by(i,j) + res_eta(i,j,0) * dtdy2 * (by(i,j+1)-2.d0*by(i,j)+by(i,j-1))  &
											  - dt/4.d0 * res_eta(i,j,1) * (bx(i,j+1)-bx(i,j-1))/dy

						elseif(kwhich==3) then

							r(i) = bz(i,j) + res_eta(i,j,0) * dtdy2 * (bz(i,j+1)-2.d0*bz(i,j)+bz(i,j-1))  &
											 + res_eta(i,j,2) * dt/4.d0 * (bz(i,j+1)-bz(i,j-1))/dy

						endif

					enddo
!!$					!$omp(enddo)

					r(0) = Bfield(1,j_p)
					r(nx+1) = Bfield(nx+2,j_p)

					! solve the tridiagonal system

					call tridiag(a,b,c,r,Btemp_i,nx+2,nx+2)
					Bnew(:,j) = Btemp_i

				enddo

!!$			!$omp enddo
!!$			!$omp end parallel

			elseif(direction(istep)==2) then
			! do diffusion in the y direction (x is treated explicitly)

!!$			!$omp parallel default(shared)
!!$			!$omp do private(i,j,a,b,c,r,i_p,j_p)

				do i = 1, nx

					i_p = i+1
					! pointer index is shifted by 1!

					! first build the coefficients

!!$					!$ omp do private(j)
					do j = 1, ny

						if(kwhich==1) then
						! Bx equation

							a(j) = -dtdy2 * res_eta(i,j,0) + res_eta(i,j,2) * dt/(4.d0*dy)

							b(j) = 1.d0 + 2.d0*dtdy2 * res_eta(i,j,0) + res_eta(i,j,0)*dt/(2.d0*Rmaj(i)**2)

							c(j) = -dtdy2 * res_eta(i,j,0) - res_eta(i,j,2) * dt/(4.d0*dy)

						elseif(kwhich==2) then
						! By equation

							a(j) = -dtdy2 * res_eta(i,j,0)

							b(j) = 1.d0 + 2.d0*dtdy2 * res_eta(i,j,0)

							c(j) = -dtdy2 * res_eta(i,j,0)

						elseif(kwhich==3) then
						! Bz equation

							a(j) = -dtdy2 * res_eta(i,j,0) + res_eta(i,j,2) * dt/(4.d0*dy)

							b(j) = 1.d0 + 2.d0*dtdy2 * res_eta(i,j,0) + res_eta(i,j,0)*dt/(2.d0*Rmaj(i)**2)  &
											+ dt/(2.d0*Rmaj(i)) * res_eta(i,j,1)

							c(j) = -dtdy2 * res_eta(i,j,0) - res_eta(i,j,2) * dt/(4.d0*dy)

						endif

					enddo
!!$					!$omp enddo

					! apply fixed boundary conditions

					a(0) = 0.d0; c(0) = 0.d0
					b(0) = 1.d0
					a(ny+1) = 0.d0; c(ny+1) = 0.d0
					b(ny+1) = 1.d0

					! build the RHS
					! for the sake of clearness, use explicit definitions

!!$					!$omp do private(j)
					do j = 1, ny

						if(kwhich==1) then

							r(j) = bx(i,j) + res_eta(i,j,0) *dtdx2 * (bx(i+1,j)-2.d0*bx(i,j)+bx(i-1,j)) +  &
												res_eta(i,j,0) * dt/(4.d0*dx*Rmaj(i)) * (bx(i+1,j)-bx(i-1,j))  &
												-dt/(4.d0*dx) * res_eta(i,j,2) * (by(i+1,j)-by(i-1,j))

						elseif(kwhich==2) then

							r(j) = by(i,j) + res_eta(i,j,0) * dtdx2 * (by(i+1,j)-2.d0*by(i,j)+by(i-1,j))  &
											  + res_eta(i,j,0) * dt/(4.d0*Rmaj(i)*dx) * (by(i+1,j)-by(i-1,j))  &
											  - dt/4.d0 * res_eta(i,j,1) * ( (bx(i,j+1)-bx(i,j-1))/dy - (by(i+1,j)-by(i-1,j))/dx )

						elseif(kwhich==3) then

							r(j) = bz(i,j) + res_eta(i,j,0) * dtdx2 * (bz(i+1,j)-2.d0*bz(i,j)+bz(i-1,j))  &
											 + res_eta(i,j,0) * dt/(4.d0*Rmaj(i)*dx) * (bz(i+1,j)-bz(i-1,j))  &
											 - res_eta(i,j,1) * dt/4.d0 * (bz(i+1,j)-bz(i-1,j))/dx

						endif

					enddo
!!$					!$omp enddo

					r(0) = Bfield(i_p,1)
					r(ny+1) = Bfield(i_p,ny+2)

					! solve the tridiagonal system

					call tridiag(a,b,c,r,Btemp_j,ny+2,ny+2)
					Bnew(i,:) = Btemp_j

				enddo

!!$			!$omp enddo
!!$			!$omp end parallel

			endif

			! update the field

!!$			!$omp parallel default(shared)
!!$			!$omp do private(i,j,a,b,c,r,i_p,j_p)
			do j = 1, ny

				j_p = j+1
				! pointer index is shifted by 1!

!!$				!$omp do private(i)
				do i = 1, nx

					i_p = i+1
					! pointer index is shifted by 1!

					Bfield(i_p,j_p) = Bnew(i,j)

				enddo
!!$				!$omp enddo

			enddo
!!$			!$omp enddo
!!$			!$omp end parallel

			which_component = cshift(which_component,1)

		enddo

		nullify(Bfield)

	enddo

        icycle_temp = icycle_temp+1

        enddo

        dt = dt * ( additional_res_steps + 1.d0 )

end subroutine resistive_diffusion_2


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine resistive_diffusion_3(icycle)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! ADI diffusion for magnetic field, curl implementation
! for the sake of clearness, equations are repeated for each component
! NOTE: some terms are always treated explicitly!

	use mod_parameters, only : nx, ny, dt, dx, dy
	use mod_arrays, only : nmx, nmy, bx, by, bz, Rmaj, Rmaj_plushalf

	integer :: icycle
	integer :: direction(2)
	real(kind=dkind), dimension(:,:), pointer :: Bfield
	real(kind=dkind), dimension(:), allocatable, save :: a, b, c, r
	real(kind=dkind), dimension(:,:), allocatable, save :: Bnew
	real(kind=dkind) :: dtdx2, dtdy2, dt8
	integer :: i, j, k, istep, kwhich
	integer :: i_p, j_p ! indexes for pointer
	integer, dimension(1:3), save :: which_component = 0

	if(allocated(a)) then
		continue
	else
		allocate(a(0:max(nx,ny)+1), b(0:max(nx,ny)+1),  &
					c(0:max(nx,ny)+1), r(0:max(nx,ny)+1),  &
					Bnew(0:nx+1,0:ny+1) )
	endif

	if(maxval(which_component)==0) then
	! set up initial values

		which_component(1) = 1
		which_component(2) = 2
		which_component(3) = 3

	endif

	nullify(Bfield)

	! half time steps
	dtdx2 = dt/dx**2 / 2.d0
	dtdy2 = dt/dy**2 / 2.d0
	dt8 = dt / (8.d0*dx*dy)

	if(modulo(icycle,2)==0) then

		direction(1) = 1
		direction(2) = 2

	elseif(modulo(icycle,2)==1) then

		direction(1) = 2
		direction(2) = 1

	endif

	do istep = 1, 2

		do k = 1, 3
		! "k" is the field component

			kwhich = which_component(k)

			if(kwhich==1) then
				Bfield => bx(0:nx+1,0:ny+1)
			elseif(kwhich==2) then
				Bfield => by(0:nx+1,0:ny+1)
			elseif(kwhich==3) then
				Bfield => bz(0:nx+1,0:ny+1)
			endif

			if(direction(istep)==1) then
			! treat x direction implicitly (where possible) and y direction explicitly

				do j = 1, ny

					j_p = j+1
					! pointer index is shifted by 1!

					! first build the coefficients

					do i = 1, nx

						if(kwhich==1) then
						! Bx equation

							a(i) = 0.d0

							b(i) = 1.d0

							c(i) = 0.d0

						elseif(kwhich==2) then
						! By equation

							a(i) = - dtdx2 * Rmaj_plushalf(i-1) * res_eta_iph(i-1,j) / Rmaj(i)

							b(i) = 1.d0 + dtdx2 / Rmaj(i) * ( res_eta_iph(i-1,j)*Rmaj_plushalf(i-1) + res_eta_iph(i,j)*Rmaj_plushalf(i) )

							c(i) = - dtdx2 * Rmaj_plushalf(i) * res_eta_iph(i,j) / Rmaj(i)

						elseif(kwhich==3) then
						! Bz equation

							a(i) = - dtdx2 * Rmaj(i-1) * res_eta_iph(i-1,j) / Rmaj_plushalf(i-1)

							b(i) = 1.d0 + dtdx2 * Rmaj(i) * ( res_eta_iph(i-1,j)/Rmaj_plushalf(i-1) + res_eta_iph(i,j)/Rmaj_plushalf(i) )

							c(i) = - dtdx2 * Rmaj(i+1) * res_eta_iph(i,j) / Rmaj_plushalf(i)

						endif

					enddo

					! apply fixed boundary conditions

					a(0) = 0.d0; c(0) = 0.d0
					b(0) = 1.d0
					a(nx+1) = 0.d0; c(nx+1) = 0.d0
					b(nx+1) = 1.d0

					! build the RHS

					do i = 1, nx

						if(kwhich==1) then

							r(i) = bx(i,j) +  &
									dtdy2 * ( res_eta_jph(i,j) * (bx(i,j+1)-bx(i,j))  &
													- res_eta_jph(i,j-1) * (bx(i,j)-bx(i,j-1))  )  &
									-dt8 * ( res_eta(i,j+1,0) * (bz(i+1,j+1)-bz(i-1,j+1)) &
												- res_eta(i,j-1,0) * (bz(i+1,j-1)-bz(i-1,j-1))  )

						elseif(kwhich==2) then

							r(i) = by(i,j)  &
									- dt8 * ( Rmaj(i+1)*res_eta(i+1,j,0) * ( bx(i+1,j+1) - bx(i+1,j-1) )  &
												- Rmaj(i-1)*res_eta(i-1,j,0) * ( bx(i-1,j+1) - bx(i-1,j-1) ) )

						elseif(kwhich==3) then

							r(i) = bz(i,j) + dtdy2 *  &
														( res_eta_jph(i,j) * ( bz(i,j+1) - bz(i,j) )  &
														  - res_eta_jph(i,j-1) * ( bz(i,j) - bz(i,j-1) ) )

						endif

					enddo

					r(0) = Bfield(1,j_p)
					r(nx+1) = Bfield(nx+2,j_p)

					! solve the tridiagonal system

					call tridiag(a,b,c,r,Bnew(:,j),nx+2,nx+2)

				enddo

			elseif(direction(istep)==2) then
			! treat y direction implicitly (where possible) and x direction explicitly

				do i = 1, nx

					i_p = i+1
					! pointer index is shifted by 1!

					! first build the coefficients

					do j = 1, ny

						if(kwhich==1) then
						! Bx equation

							a(j) = -dtdy2 * res_eta_jph(i,j-1)

							b(j) = 1.d0 + dtdy2 * (res_eta_jph(i,j-1)+res_eta_jph(i,j))

							c(j) = -dtdy2 * res_eta_jph(i,j)

						elseif(kwhich==2) then
						! By equation

							a(i) = 0.d0

							b(i) = 1.d0

							c(i) = 0.d0

						elseif(kwhich==3) then
						! Bz equation

							a(j) = -dtdy2 * res_eta_jph(i,j-1)

							b(j) = 1.d0 + dtdy2 * (res_eta_jph(i,j-1)+res_eta_jph(i,j))

							c(j) = -dtdy2 * res_eta_jph(i,j)

						endif

					enddo

					! apply fixed boundary conditions

					a(0) = 0.d0; c(0) = 0.d0
					b(0) = 1.d0
					a(ny+1) = 0.d0; c(ny+1) = 0.d0
					b(ny+1) = 1.d0

					! build the RHS

					do j = 1, ny

						if(kwhich==1) then

							r(j) = bx(i,j) +  &
												dt8 * ( res_eta(i,j+1,0) * ( bz(i+1,j+1) - bz(i-1,j+1) )  &
															- res_eta(i,j-1,0) * ( bz(i+1,j-1) - bz(i-1,j+1) ) )

						elseif(kwhich==2) then

							r(j) = by(i,j) +  &
									dtdx2/Rmaj(i) * ( res_eta_iph(i,j)*Rmaj_plushalf(i) * (by(i+1,j)-by(i,j))  &
															  - res_eta_iph(i-1,j)*Rmaj_plushalf(i-1) * (by(i,j)-by(i-1,j)) )  &
									-dt8/Rmaj(i) * ( Rmaj(i+1)*res_eta(i+1,j,0) * ( bx(i+1,j+1) - bx(i+1,j-1) )  &
															- Rmaj(i-1)*res_eta(i-1,j,0) * ( bx(i-1,j+1) - bx(i-1,j-1) ) )

						elseif(kwhich==3) then

							r(j) = bz(i,j) +  &
									dtdx2 * ( res_eta_iph(i,j) * ( Rmaj(i+1)*bz(i+1,j) - Rmaj(i)*bz(i,j) ) / Rmaj_plushalf(i)  &
													-res_eta_iph(i-1,j) * ( Rmaj(i)*bz(i,j) - Rmaj(i-1)*bz(i-1,j) ) / Rmaj_plushalf(i-1) )

						endif

					enddo

					r(0) = Bfield(i_p,1)
					r(ny+1) = Bfield(i_p,ny+2)

					! solve the tridiagonal system

					call tridiag(a,b,c,r,Bnew(i,:),ny+2,ny+2)

				enddo

			endif

			! update the field

			do j = 1, ny

				j_p = j+1
				! pointer index is shifted by 1!

				do i = 1, nx

					i_p = i+1
					! pointer index is shifted by 1!

					Bfield(i_p,j_p) = Bnew(i,j)

				enddo

			enddo

			which_component = cshift(which_component,1)

		enddo

		nullify(Bfield)

	enddo

end subroutine resistive_diffusion_3


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine resistive_diffusion_4(icycle)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! ADI diffusion for magnetic field
! for the sake of clearness, equations are repeated for each component
! thie routine uses the time-evolved resistivity

	use mod_parameters, only : nx, ny, dt, dx, dy
	use mod_arrays, only : nmx, nmy, bx, by, bz, Rmaj

	integer :: icycle
	integer :: direction(2)
	real(kind=dkind), dimension(:,:), pointer :: Bfield
	real(kind=dkind), dimension(:), allocatable, save :: a, b, c, r
	real(kind=dkind), dimension(:,:), allocatable, save :: Bnew
	real(kind=dkind) :: dtdx2, dtdy2
	integer :: i, j, k, istep, kwhich
	integer :: i_p, j_p ! indexes for pointer
	integer, dimension(1:3), save :: which_component = 0

	if(allocated(a)) then
		continue
	else
		allocate(a(0:max(nx,ny)+1), b(0:max(nx,ny)+1),  &
					c(0:max(nx,ny)+1), r(0:max(nx,ny)+1),  &
					Bnew(0:nx+1,0:ny+1) )
	endif

	if(maxval(which_component)==0) then
	! set up initial values

		which_component(1) = 1
		which_component(2) = 2
		which_component(3) = 3

	endif

	nullify(Bfield)

	! half time steps
	dtdx2 = dt/dx**2 / 2.d0
	dtdy2 = dt/dy**2 / 2.d0

	if(modulo(icycle,2)==0) then

		direction(1) = 1
		direction(2) = 2

	elseif(modulo(icycle,2)==1) then

		direction(1) = 2
		direction(2) = 1

	endif

	do istep = 1, 2

		do k = 1, 3
		! "k" is the field component

			kwhich = which_component(k)

			if(kwhich==1) then
				Bfield => bx(0:nx+1,0:ny+1)
			elseif(kwhich==2) then
				Bfield => by(0:nx+1,0:ny+1)
			elseif(kwhich==3) then
				Bfield => bz(0:nx+1,0:ny+1)
			endif

!			if(kwhich/=3) cycle

			if(direction(istep)==1) then
			! do diffusion in the x direction (y is treated explicitly)

				do j = 1, ny

					j_p = j+1
					! pointer index is shifted by 1!

					! first build the coefficients

					do i = 1, nx

						if(kwhich==1) then
						! Bx equation

							a(i) = -res_eta(i,j,0) * dtdx2 * ( 1.d0 - dx/(2.d0*Rmaj(i)) )

							b(i) = 1.d0 + res_eta(i,j,0) * (2.d0*dtdx2 +  dt/(2.d0*Rmaj(i)**2) )

							c(i) = -res_eta(i,j,0) * dtdx2 * ( 1.d0 + dx/(2.d0*Rmaj(i)) )

						elseif(kwhich==2) then
						! By equation

							a(i) = -res_eta(i,j,0) * dtdx2 * ( 1.d0 - dx/(2.d0*Rmaj(i)) )  &
										+(res_eta(i+1,j,0)-res_eta(i-1,j,0))/(2.d0*dx) * dt/(4.d0*dx)

							b(i) = 1.d0 + res_eta(i,j,0) * 2.d0*dtdx2

							c(i) = -res_eta(i,j,0) * dtdx2 * ( 1.d0 + dx/(2.d0*Rmaj(i)) )  &
										-(res_eta(i+1,j,0)-res_eta(i-1,j,0))/(2.d0*dx) * dt/(4.d0*dx)

						elseif(kwhich==3) then
						! Bz equation

							a(i) = -res_eta(i,j,0) * dtdx2 * ( 1.d0 - dx/(2.d0*Rmaj(i)) )  &
										-(res_eta(i+1,j,0)-res_eta(i-1,j,0))/(2.d0*dx) * dt/(4.d0*dx)

							b(i) = 1.d0 + res_eta(i,j,0) * (2.d0*dtdx2 +  dt/(2.d0*Rmaj(i)**2) )  &
												+(res_eta(i+1,j,0)-res_eta(i-1,j,0))/(2.d0*dx) * dt/(2.d0*Rmaj(i))

							c(i) = -res_eta(i,j,0) * dtdx2 * ( 1.d0 + dx/(2.d0*Rmaj(i)) )  &
										+(res_eta(i+1,j,0)-res_eta(i-1,j,0))/(2.d0*dx)* dt/(4.d0*dx)

						endif

					enddo

					! apply fixed boundary conditions

					a(0) = 0.d0; c(0) = 0.d0
					b(0) = 1.d0
					a(nx+1) = 0.d0; c(nx+1) = 0.d0
					b(nx+1) = 1.d0

					! build the RHS
					! for the sake of clearness, use explicit definitions

					do i = 1, nx

						if(kwhich==1) then

							r(i) = bx(i,j) + res_eta(i,j,0) *dtdy2 * (bx(i,j+1)-2.d0*bx(i,j)+bx(i,j-1)) +  &
												dt/4.d0 * (res_eta(i,j+1,0)-res_eta(i,j-1,0))/(2.d0*dy)  &
															* ( (bx(i,j+1)-bx(i,j-1))/dy - (by(i+1,j)-by(i-1,j))/dx )

						elseif(kwhich==2) then

							r(i) = by(i,j) + res_eta(i,j,0) * dtdy2 * (by(i,j+1)-2.d0*by(i,j)+by(i,j-1))  &
											  - dt/4.d0 * (res_eta(i+1,j,0)-res_eta(i-1,j,0))/(2.d0*dx) * (bx(i,j+1)-bx(i,j-1))/dy

						elseif(kwhich==3) then

							r(i) = bz(i,j) + res_eta(i,j,0) * dtdy2 * (bz(i,j+1)-2.d0*bz(i,j)+bz(i,j-1))  &
											 + (res_eta(i,j+1,0)-res_eta(i,j-1,0))/(2.d0*dy) * dt/4.d0 * (bz(i,j+1)-bz(i,j-1))/dy

						endif

					enddo

					r(0) = Bfield(1,j_p)
					r(nx+1) = Bfield(nx+2,j_p)

					! solve the tridiagonal system

					call tridiag(a,b,c,r,Bnew(:,j),nx+2,nx+2)

				enddo

			elseif(direction(istep)==2) then
			! do diffusion in the y direction (x is treated explicitly)

				do i = 1, nx

					i_p = i+1
					! pointer index is shifted by 1!

					! first build the coefficients

					do j = 1, ny

						if(kwhich==1) then
						! Bx equation

							a(j) = -dtdy2 * res_eta(i,j,0) + (res_eta(i,j+1,0)-res_eta(i,j-1,0))/(2.d0*dy) * dt/(4.d0*dy)

							b(j) = 1.d0 + 2.d0*dtdy2 * res_eta(i,j,0) + res_eta(i,j,0)*dt/(2.d0*Rmaj(i)**2)

							c(j) = -dtdy2 * res_eta(i,j,0) - (res_eta(i,j+1,0)-res_eta(i,j-1,0))/(2.d0*dy) * dt/(4.d0*dy)

						elseif(kwhich==2) then
						! By equation

							a(j) = -dtdy2 * res_eta(i,j,0)

							b(j) = 1.d0 + 2.d0*dtdy2 * res_eta(i,j,0)

							c(j) = -dtdy2 * res_eta(i,j,0)

						elseif(kwhich==3) then
						! Bz equation

							a(j) = -dtdy2 * res_eta(i,j,0) + (res_eta(i,j+1,0)-res_eta(i,j-1,0))/(2.d0*dy) * dt/(4.d0*dy)

							b(j) = 1.d0 + 2.d0*dtdy2 * res_eta(i,j,0) + res_eta(i,j,0)*dt/(2.d0*Rmaj(i)**2)  &
											+ dt/(2.d0*Rmaj(i)) * (res_eta(i+1,j,0)-res_eta(i-1,j,0))/(2.d0*dx)

							c(j) = -dtdy2 * res_eta(i,j,0) - (res_eta(i,j+1,0)-res_eta(i,j-1,0))/(2.d0*dy) * dt/(4.d0*dy)

						endif

					enddo

					! apply fixed boundary conditions

					a(0) = 0.d0; c(0) = 0.d0
					b(0) = 1.d0
					a(ny+1) = 0.d0; c(ny+1) = 0.d0
					b(ny+1) = 1.d0

					! build the RHS
					! for the sake of clearness, use explicit definitions

					do j = 1, ny

						if(kwhich==1) then

							r(j) = bx(i,j) + res_eta(i,j,0) *dtdx2 * (bx(i+1,j)-2.d0*bx(i,j)+bx(i-1,j)) +  &
												res_eta(i,j,0) * dt/(4.d0*dx*Rmaj(i)) * (bx(i+1,j)-bx(i-1,j))  &
												-dt/(4.d0*dx) * (res_eta(i,j+1,0)-res_eta(i,j-1,0))/(2.d0*dy) * (by(i+1,j)-by(i-1,j))

						elseif(kwhich==2) then

							r(j) = by(i,j) + res_eta(i,j,0) * dtdx2 * (by(i+1,j)-2.d0*by(i,j)+by(i-1,j))  &
											  + res_eta(i,j,0) * dt/(4.d0*Rmaj(i)*dx) * (by(i+1,j)-by(i-1,j))  &
											  - dt/4.d0 * (res_eta(i+1,j,0)-res_eta(i-1,j,0))/(2.d0*dx) * ( (bx(i,j+1)-bx(i,j-1))/dy - (by(i+1,j)-by(i-1,j))/dx )

						elseif(kwhich==3) then

							r(j) = bz(i,j) + res_eta(i,j,0) * dtdx2 * (bz(i+1,j)-2.d0*bz(i,j)+bz(i-1,j))  &
											 + res_eta(i,j,0) * dt/(4.d0*Rmaj(i)*dx) * (bz(i+1,j)-bz(i-1,j))  &
											 - (res_eta(i+1,j,0)-res_eta(i-1,j,0))/(2.d0*dx) * dt/4.d0 * (bz(i+1,j)-bz(i-1,j))/dx

						endif

					enddo

					r(0) = Bfield(i_p,1)
					r(ny+1) = Bfield(i_p,ny+2)

					! solve the tridiagonal system

					call tridiag(a,b,c,r,Bnew(i,:),ny+2,ny+2)

				enddo

			endif

			! update the field

			do j = 1, ny

				j_p = j+1
				! pointer index is shifted by 1!

				do i = 1, nx

					i_p = i+1
					! pointer index is shifted by 1!

					Bfield(i_p,j_p) = Bnew(i,j)

				enddo

			enddo

			which_component = cshift(which_component,1)

		enddo

		nullify(Bfield)

	enddo

end subroutine resistive_diffusion_4


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  
subroutine radius(i,j,ex,ey,r)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  
! this subroutine computes the minor radius r of the plasma
! as a function of the grid indexes (i,j)

	use mod_arrays, only : xV, yV
	use mod_parameters, only : pi, nx, ny, xlength, ylength
	use pseudo_IMSL, only : dbsval

	integer, intent(in) :: i,j
	real (kind=dkind), intent(out) :: ex,ey,r
	real (kind=dkind) :: angle


	if( (i==0).or.(j==0).or.(i>nx).or.(j>ny) ) then
	! somehow the code got here with an index exceeding outside the computation region
	! assign this to be an external point

		ex = 1.d0
		ey = 1.d0
		r = 1.d-2
		return

	endif

	ex = xV(i)-xlength/2.d0
	ey = yV(j)-ylength/2.d0

	if (ex==0.d0) then
		angle = pi/2.d0 * dsign(1.d0,ey)
	else
		angle = datan2(ey,ex)
	endif

	r = 0.d0

	if(angle>=r_data(theta_points,1)) then

		angle = angle - 2.d0*pi

	elseif(angle<r_data(1,1)) then

		angle = angle + 2.d0*pi

	endif

	r = dbsval(angle, r_ord, r_data(:,3),  &
				theta_points, r_cscoef(1,1:theta_points) )


	continue

end subroutine radius



!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  
subroutine radius_der(x,y,ex,ey,angle,r,rprim,rsec)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  

	use mod_parameters, only : pi, rmajor
	use pseudo_IMSL, only : dbsval, dbsder

	real (kind=dkind), intent(in) :: x, y
	real (kind=dkind), intent(out) :: r
	real (kind=dkind) :: ex,ey, angle
	real(kind=dkind) ::  rprim, rsec

	ex = x - rmajor
	ey = y

	if (ex==0.d0) then
		angle = pi/2.d0 * dsign(1.d0,ey)
	else
		angle = datan2(ey,ex)
	endif

	r = 0.d0

	if(angle>=r_data(theta_points,1)) then

		angle = angle - 2.d0*pi

	elseif(angle<r_data(1,1)) then

		angle = angle + 2.d0*pi

	endif

	r = dbsval(angle, r_ord, r_data(:,3),  &
		theta_points, r_cscoef(1,1:theta_points) )

	rprim = dbsder(1,angle, r_ord, r_data(:,3),  &
			theta_points, r_cscoef(1,1:theta_points) )

	rsec = dbsder(2,angle, r_ord, r_data(:,3),  &
			theta_points, r_cscoef(1,1:theta_points) )


	continue

end subroutine radius_der


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  
subroutine radius_theta(angle,r,x,z)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  
	! this subroutine computes the minor radius r of the plasma
	! as a function of the angle; x and z are also returned

	use mod_parameters, only : pi, rmajor
	use pseudo_IMSL, only : dbsval

	real (kind=dkind) :: r, x, z
	integer k
	real(kind=dkind) :: angle, theta

	r = 0.d0
	theta = angle

	if(theta>=r_data(theta_points,1)) then

		theta = theta - 2.d0*pi

	elseif(theta<r_data(1,1)) then

		theta = theta + 2.d0*pi

	endif

	r = dbsval(theta, r_ord, r_data(:,3),  &
		theta_points, r_cscoef(1,1:theta_points) )


	x = rmajor + r*cos(angle)
	z = r*sin(angle)

	continue

end subroutine radius_theta


end module vacuum_module
