module numerical_input

	implicit none

	private

	 public numerical_equilibrium, FLOW_equilibrium, read_numerical,  &
				ibreak_p, ibreak_th, ibreak_d, ibreak_Bth, ibreak_Bz,  &
				numerical_n, numerical_p, numerical_Bz, numerical_mtheta,  &
				p_ord, d_ord, Bz_ord,  d_data, d_cscoef,  &
				p_data, p_cscoef, Bz_data, Bz_cscoef

	integer, parameter, private :: dkind = kind(1.d0)
	integer, parameter, private :: data_dim_lim = 2000
	
	integer :: data_dim

	integer :: ibreak_p, ibreak_th, ibreak_d, ibreak_Bth, ibreak_Bz

	real(kind=dkind) p_data(data_dim_lim,3)	!numerical data for p(r)
	real(kind=dkind) d_data(data_dim_lim,3)	!numerical data for rho(r)
	real(kind=dkind) mtheta_data(data_dim_lim,3)	!numerical data for Mtheta(r)
	real(kind=dkind) Bz_data(data_dim_lim,3)	!numerical data for Bz(r)
	real(kind=dkind) Bth_data(data_dim_lim,3)	!numerical data for Bth(r)

	logical :: numerical_n = .true.		!whether to use numerical data
	logical :: numerical_p = .true.		!whether to use numerical data
	logical :: numerical_mtheta = .true.		!whether to use numerical data
	logical :: numerical_Bth, numerical_Bz
	logical :: proceed = .false.

	real(kind=dkind), dimension(:,:), allocatable :: p_cscoef,   &
															mtheta_cscoef,  d_cscoef, Bz_cscoef, Bth_cscoef
	! for IMSL spline routine

	integer :: p_ord = 2	! interpolation order
	integer :: mtheta_ord = 2	! interpolation order
	integer :: d_ord = 2		! interpolation order
	integer :: Bz_ord = 2		! interpolation order
	integer :: Bth_ord = 2		! interpolation order

	contains


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  
subroutine numerical_equilibrium
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  

	use mod_parameters,  ONLY: rsh,pi, nx,ny,dx,dy,xlength,ylength

    use mod_arrays,      ONLY:nmx,nmy,rho,bx, by, bz,u,v,w,p,Tn, xV,yV

	real(kind=dkind) :: rloc, thetaloc, rholoc, ploc, vloc, Bzloc, Bthloc
	! local values of r, rho, p, v

    real(kind=dkind) :: tv(0:nmy+1),    vv(0:nmy+1),     &
              rhov(0:nmy+1),  pv(0:nmy+1),     &
			  p_old(0:nmy+1), ptot(0:nmy+1), phi(0:nmy+1)

	integer :: i,j


	call read_numerical

	do j=0,ny+1
	do i=0,nx+1

! Set B equal to zero everywhere.
	bx(i,j)	=	0
	by(i,j)	=	0
	bz(i,j)	=	0

	rloc = sqrt((xV(i)-0.5d0)**2+(yV(j)-0.5d0)**2)

	if((xV(i)-xlength/2.d0)*(yV(j)-ylength/2.d0)==0.d0) then
		thetaloc = 0.d0	! atan2 does not work with 0 input, but for r=0 the angle won't matter
	else
		thetaloc = datan2((xV(i)-xlength/2.d0),(yV(j)-ylength/2.d0))
!		thetaloc = datan2((yV(j)-0.5d0),(xV(i)-0.5d0))
	endif

	call numerical_input_interpolation(rloc,rholoc,ploc,vloc, Bzloc, Bthloc)

	p(i,j)  =  ploc
	rho(i,j)  =  rholoc
	v(i,j)  =  vloc*sin(thetaloc)
	u(i,j)  =  -vloc*cos(thetaloc)
	Tn(i,j) = ploc/rholoc
	w(i,j) = 0.d0
	bz(i,j) = Bzloc
	bx(i,j) = -Bthloc*cos(thetaloc)
	by(i,j)  =  Bthloc*sin(thetaloc)

	end do
	end do	




	call input_cleanup


end subroutine numerical_equilibrium

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  
subroutine read_numerical
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  

	integer i
	integer :: d_dim

	if(numerical_n) then

		i = 0

		open(33,file='input/n.dat', status='old', action='read')

		do

			i = i+1
			read(33,*,end=19) d_data(i,1), d_data(i,2)

		enddo

	19	close(33)

		d_dim = i-1
		ibreak_d = d_dim + d_ord

		allocate(d_cscoef(1,d_dim))

		call interp_setup(d_dim,d_ord, &
			d_data(1:d_dim,1),d_data(1:d_dim,2), &
			d_data(1:ibreak_d,3),d_cscoef(1,1:d_dim))


	endif


!----------------------------------------------------------------------------

	if(numerical_p) then

	i = 0

	open(33,file='input/p.dat', status='old', action='read')

	do 

		i = i+1
		read(33,*,end=66) p_data(i,1), p_data(i,2)

	enddo

66	close(33)

	data_dim = i-1
	ibreak_p = data_dim + p_ord

	allocate(p_cscoef(1,data_dim))

	call interp_setup(data_dim,p_ord, &
		p_data(1:data_dim,1),p_data(1:data_dim,2), &
		p_data(1:ibreak_p,3),p_cscoef(1,1:data_dim))


	endif


!----------------------------------------------------------------------------

	if(numerical_mtheta) then

		open(33,file='input/mtheta.dat', status='old', action='read')

		i = 0

		do

			i = i+1

			read(33,*,end=109) mtheta_data(i,1),mtheta_data(i,2)

		enddo

		109	close(33)

		data_dim = i-1
		ibreak_th = data_dim + mtheta_ord

		allocate(mtheta_cscoef(1,data_dim))

		! interpolation setup has been moved to a separate function
		call interp_setup(data_dim,mtheta_ord, &
			mtheta_data(1:data_dim,1),mtheta_data(1:data_dim,2), &
			mtheta_data(1:ibreak_th,3),mtheta_cscoef(1,1:data_dim))

	endif


!----------------------------------------------------------------------------


	inquire(file='input/Bth.dat',exist=proceed)

	if(proceed) then

		numerical_Bth = .true.

	endif

	if(numerical_Bth) then

	i = 0

	open(33,file='input/Bth.dat', status='old', action='read')

	do 

		i = i+1
		read(33,*,end=201) Bth_data(i,1), Bth_data(i,2)

	enddo

201	close(33)

	data_dim = i-1
	ibreak_Bth = data_dim + Bth_ord

	allocate(Bth_cscoef(1,data_dim))

	call interp_setup(data_dim,Bth_ord, &
		Bth_data(1:data_dim,1),Bth_data(1:data_dim,2), &
		Bth_data(1:ibreak_Bth,3),Bth_cscoef(1,1:data_dim))


	endif


!----------------------------------------------------------------------------


	inquire(file='input/Bz.dat',exist=proceed)

	if(proceed) then

		numerical_Bz = .true.

	endif

	if(numerical_Bz) then

	i = 0

	open(33,file='input/Bz.dat', status='old', action='read')

	do 

		i = i+1
		read(33,*,end=211) Bz_data(i,1), Bz_data(i,2)

	enddo

211	close(33)

	data_dim = i-1
	ibreak_Bz = data_dim + Bz_ord

	allocate(Bz_cscoef(1,data_dim))

	call interp_setup(data_dim,Bz_ord, &
		Bz_data(1:data_dim,1),Bz_data(1:data_dim,2), &
		Bz_data(1:ibreak_Bz,3),Bz_cscoef(1,1:data_dim))


	endif

!----------------------------------------------------------------------------


	continue

end subroutine read_numerical

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine interp_setup(npoints,int_ord,xdata,ydata,data3,cscoef)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	use pseudo_IMSL, only : DBSNAK, DBSINT

	implicit none

	integer :: int_ord	! order of the interpolation
	integer :: npoints	! dimension of the input data

	real(kind=dkind) :: xdata(npoints)	! the "x" data
	real(kind=dkind) :: ydata(npoints)	! the "y" data
	real(kind=dkind) :: data3(npoints+int_ord)	! the "x" points in the interpolation
	real(kind=dkind) :: cscoef(npoints)	! the interpolation coefficients

	call DBSNAK(npoints, xdata, int_ord,data3)

	call DBSINT(npoints, xdata, ydata, int_ord, data3, cscoef)

	continue

	return

end subroutine interp_setup


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine numerical_input_interpolation(etaloc,rholoc,ploc,vloc, Bzloc, Bthloc)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	use pseudo_IMSL, only : dbsval
	use mod_parameters, only : rsh

	real (kind=dkind) :: etaloc, rholoc, ploc, vloc, Bzloc, Bthloc
	real (kind=dkind) :: mtheta

	ploc = dbsval(etaloc, p_ord, p_data(1:ibreak_p,3),  &
			ibreak_p-p_ord, p_cscoef(1,1:ibreak_p-p_ord) )

	rholoc = dbsval(etaloc, d_ord, d_data(1:ibreak_d,3),  &
			ibreak_d-d_ord, d_cscoef(1,1:ibreak_d-d_ord) )

	mtheta =  dbsval(etaloc, mtheta_ord, mtheta_data(1:ibreak_th,3),  &
			ibreak_th-mtheta_ord, mtheta_cscoef(1,1:ibreak_th-mtheta_ord) )

	vloc = sqrt(rsh*ploc/rholoc) * mtheta

	if(numerical_Bz) then
		Bzloc = dbsval(etaloc, Bz_ord, Bz_data(1:ibreak_Bz,3),  &
			ibreak_Bz-Bz_ord, Bz_cscoef(1,1:ibreak_Bz-Bz_ord) )
	else
		Bzloc = 0.d0
	endif

	if(numerical_Bth) then
		Bthloc = dbsval(etaloc, Bth_ord, Bth_data(1:ibreak_Bth,3),  &
			ibreak_Bth-Bth_ord, Bth_cscoef(1,1:ibreak_Bth-Bth_ord) )
	else
		Bthloc = 0.d0
	endif

!	write(file_69,*) etaloc, rholoc, ploc, vloc, mtheta

end subroutine numerical_input_interpolation


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  
subroutine FLOW_equilibrium
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  

	use mod_parameters,  ONLY: nx,ny,pmin,rhomin

    use mod_arrays,      ONLY: rho, bx, by, bz, u, v, w, p, Tn, xV,yV

	real(kind=dkind) :: xtemp, ytemp, error ! for debugging purposes

	integer :: i_rho, i_p, i_u, i_v, i_w, i_bx, i_by, i_bz != [31/32/33/34/35/36/37/38]
	integer :: i, j

	i_rho = 31
	i_p = 32
	i_u = 33
	i_v = 34
	i_w = 35
	i_bx = 36
	i_by = 37
	i_bz = 38


	!------------------------rho------------------------
	open(i_rho, file='input/rho_ART.dat', status='old', action = 'read')
	open(69, file='input/rho_ART_read.plt', action = 'write')

	write(69,*)'TITLE="input rho" '
	write(69,*)'Variables = "X", "Y","RHO"'
	write(69,*)'ZONE I=',nx+2,',J=',ny+2,',F=Point'

	do i=0,nx+1
	do j=0,ny+1

		read(i_rho,44) xtemp, ytemp, rho(i,j)
		rho(i,j) = max(rho(i,j), pmin)
		write(69,*) xv(i), yv(j), rho(i,j)

	end do
	end do

	close(i_rho)
	close(69)

	!------------------------p------------------------
	open(i_p, file='input/p_ART.dat', status='old', action = 'read')
	open(69, file='input/p_ART_read.plt', action = 'write')

	write(69,*)'TITLE="input p"'
	write(69,*)'Variables = "X", "Y","p"'
	write(69,*)'ZONE I=',nx+2,',J=',ny+2,',F=Point'

	do i=0,nx+1
	do j=0,ny+1

		read(i_p,44) xtemp, ytemp, p(i,j)
		p(i,j) = max(p(i,j), pmin)
		write(69,*) xv(i), yv(j), p(i,j)

	end do
	end do

	close(i_p)
	close(69)

	!------------------------u------------------------
	open(i_u, file='input/u_ART.dat', status='old', action = 'read')
	open(69, file='input/u_ART_read.plt', action = 'write')

	write(69,*)'TITLE="input u"'
	write(69,*)'Variables = "X", "Y","u"'
	write(69,*)'ZONE I=',nx+2,',J=',ny+2,',F=Point'

	do i=0,nx+1
	do j=0,ny+1

		read(i_u,44) xtemp, ytemp, u(i,j)
		write(69,*) xv(i), yv(j), u(i,j)

	end do
	end do

	close(i_u)
	close(69)

	!------------------------v------------------------
	open(i_v, file='input/v_ART.dat', status='old', action = 'read')
	open(69, file='input/v_ART_read.plt', action = 'write')

	write(69,*)'TITLE="input v"'
	write(69,*)'Variables = "X", "Y","v"'
	write(69,*)'ZONE I=',nx+2,',J=',ny+2,',F=Point'

	do i=0,nx+1
	do j=0,ny+1

		read(i_v,44) xtemp, ytemp, v(i,j)
		write(69,*) xv(i), yv(j), v(i,j)

	end do
	end do

	close(i_v)
	close(69)

	!------------------------w------------------------
	open(i_w, file='input/w_ART.dat', status='old', action = 'read')
	open(69, file='input/w_ART_read.plt', action = 'write')

	write(69,*)'TITLE="input w"'
	write(69,*)'Variables = "X", "Y","w"'
	write(69,*)'ZONE I=',nx+2,',J=',ny+2,',F=Point'

	do i=0,nx+1
	do j=0,ny+1

		read(i_w,44) xtemp, ytemp, w(i,j)
		write(69,*) xv(i), yv(j), w(i,j)

	end do
	end do

	close(i_w)
	close(69)

	!------------------------bx------------------------
	open(i_bx, file='input/Bx_ART.dat', status='old', action = 'read')
	open(69, file='input/bx_ART_read.plt', action = 'write')

	write(69,*)'TITLE="input bx"'
	write(69,*)'Variables = "X", "Y","bx"'
	write(69,*)'ZONE I=',nx+2,',J=',ny+2,',F=Point'

	do i=0,nx+1
	do j=0,ny+1

		read(i_bx,44) xtemp, ytemp, bx(i,j)
		write(69,*) xv(i), yv(j), bx(i,j)

	end do
	end do

	close(i_bx)
	close(69)

	!------------------------by------------------------
	open(i_by, file='input/By_ART.dat', status='old', action = 'read')
	open(69, file='input/by_ART_read.plt', action = 'write')

	write(69,*)'TITLE="input by"'
	write(69,*)'Variables = "X", "Y","by"'
	write(69,*)'ZONE I=',nx+2,',J=',ny+2,',F=Point'

	do i=0,nx+1
	do j=0,ny+1

		read(i_by,44) xtemp, ytemp, by(i,j)
		write(69,*) xv(i), yv(j), by(i,j)

	end do
	end do

	close(i_by)
	close(69)

	!------------------------bz------------------------
	open(i_bz, file='input/Bz_ART.dat', status='old', action = 'read')
	open(69, file='input/bz_ART_read.plt', action = 'write')

	write(69,*)'TITLE="input bz"'
	write(69,*)'Variables = "X", "Y","bz"'
	write(69,*)'ZONE I=',nx+2,',J=',ny+2,',F=Point'

	do i=0,nx+1
	do j=0,ny+1

		read(i_bz,44) xtemp, ytemp, bz(i,j)
		write(69,*) xv(i), yv(j), bz(i,j)

	end do
	end do

	close(i_bz)
	close(69)

	!------------------------T------------------------
	do i=0,nx+1
	do j=0,ny+1

		Tn(i,j) = p(i,j)/rho(i,j)

	end do
	end do	


44 format(3(e18.12, 2x))

	open(69, file='input/0_read.plt', action = 'write')

	write(69,*)'TITLE="solution at time t= 0- "'
	write(69,*)'Variables = "X", "Y","RHO","BX","BY","BZ","P","U","V","W"'
	write(69,*)'ZONE I=',nx+2,',J=',ny+2,',F=Point'

	do j = 0,ny+1
	do i = 0,nx+1

		write(69,222)xV(i),yV(j),rho(i,j),bx(i,j),by(i,j),bz(i,j),p(i,j),u(i,j),v(i,j),w(i,j)

	enddo
	enddo

	close(69)

 222 format(E13.6,10(5x,E13.6))

	continue

	return

end subroutine FLOW_equilibrium


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine input_cleanup
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	if(allocated(p_cscoef)) deallocate(p_cscoef)
	if(allocated(mtheta_cscoef)) deallocate(mtheta_cscoef)
	if(allocated(d_cscoef)) deallocate(d_cscoef)
	if(allocated(Bz_cscoef)) deallocate(Bz_cscoef)
	if(allocated(Bth_cscoef)) deallocate(Bth_cscoef)

end subroutine input_cleanup




end module numerical_input
