module GS_equilibrium

	use mod_parameters, only : dkind, rmajor, torus, rmu0, FV, FC, psi_axis
	use mod_arrays, only : psi_init
	use numerical_input, only : read_numerical, numerical_p, numerical_Bz,  &
												 p_ord, p_data, ibreak_p, p_cscoef,  &
												 Bz_ord, Bz_data, ibreak_Bz, Bz_cscoef,  &
												 numerical_n, d_ord, d_data, d_cscoef,  &
												 ibreak_d
	use pseudo_IMSL, only : dbsval, dbsder

	implicit none

!	private
!	public FLOW,  &
!				GS_solution, dofpsi, pofpsi,  &
!				rmajor_FLOW, dcenter, de_o_dc, alpha, alpha_rho, beta_center,  &
!				pe_o_pc, p_add, b_phi_zero, Fc_o_Fv, kappa, mach_phi_max,  &
!				alpha_mphi, mphi_min, mach_theta_max, mach_theta_edge, t_mth,  &
!				mach_theta_option

	integer, parameter :: skind = kind(1.e0)

	real(kind=dkind) :: psic	! axis value of psi
	real(kind=dkind) :: alpha = 2.d0	! pressure equation exponent  !2.d0
	real(kind=dkind) :: kappa = 3.d0	! F(psi) equation exponent   !3.d0
	real(kind=dkind) :: beta_center = 1.d-1
	real(kind=dkind) :: pcenter, pedge
!	real(kind=dkind) :: pe_o_pc = 2.d-2	! pedge/pcenter
	real(kind=dkind) :: b_phi_zero = 1.d0
!	real(kind=dkind) :: FV, FC
	real(kind=dkind) :: deltaF = 0.2d0	! FC=FV*(1+deltaF)

	real(kind=dkind) :: dcenter, dedge, alpha_rho
	real(kind=dkind) :: mach_theta_max = 2.d0
	real(kind=dkind) :: mach_theta_edge = 5.d0
	real(kind=dkind) :: t_mth = 0.5d0
	real(kind=dkind) :: mach_phi_max, mphi_min, alpha_mphi
	real(kind=dkind) :: p_add = 0.d0

	integer :: mach_theta_option = 5

	real(kind=dkind) :: rmajor_FLOW
	real(kind=dkind) :: de_o_dc, pe_o_pc, Fc_o_Fv



	real(kind=dkind), allocatable, dimension(:,:) :: psi

	namelist/FLOW/ rmajor_FLOW , dcenter, de_o_dc, alpha, alpha_rho, beta_center,  &
							pe_o_pc, p_add, b_phi_zero, Fc_o_Fv, kappa, mach_phi_max,  &
							alpha_mphi, mphi_min, &
							mach_theta_max, mach_theta_edge, t_mth, mach_theta_option


	contains

	!-----------------------------------

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  
subroutine GS_solution
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  

	use mod_arrays, only : Rmaj
	use mod_parameters, only : numerical_input_option

	logical :: toroidal = .true.

	open(unit=5,file='input/inputfile_SIM2D.dat',status='old', action='read')
	read(5,FLOW)
	close(5)

!	if(torus==0.d0) toroidal = .false.

!	torus = 1.d0

	if(abs(numerical_input_option)==7) then

		call read_GS_FLOW
		return

	elseif((numerical_input_option==8).or.(numerical_input_option==9)) then

		call read_GS_FLOW_all_B
		return

	endif

!	if(toroidal) then
!		continue
!	else
!		torus = 0.d0
!		Rmaj = rmajor
!	endif

	continue

	return

end subroutine GS_solution


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine read_GS_FLOW
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! this routine reads FLOW output and interpolates to create the input
! note the difference in notation y <-> z

	use mod_parameters, only : rsh, nx, ny, rhomin, xlength, ylength,  &
												numerical_input_option, rho_boundary, p_boundary
	use mod_arrays, only : rho, bx, by, bz, u, v, w, p, Rmaj, xV, yV
	use pseudo_IMSL, only :DBSNAK, DBS2IN, DBS2VL

	integer :: i, j
	character :: dummy_char
	real(kind=dkind) :: dummy
	integer :: n_FLOW
	real(kind=dkind), dimension(:,:), allocatable :: psi_FLOW, Bx_FLOW, By_FLOW
	real(kind=dkind), dimension(:), allocatable :: x_FLOW, y_FLOW
	real(kind=dkind) :: dx2_FLOW, dy2_FLOW, Rloc
	real(kind=dkind) :: psiloc, xloc, yloc, Bxloc, Byloc, phi_rho, omloc
	real(kind=dkind), dimension(:), allocatable :: xknot, yknot
	real(kind=dkind), dimension(:,:), allocatable :: psi_bscoef, Bx_bscoef, By_bscoef
	integer :: FLOW_ord = 2	! interpolation order

	! initializations




	if(rmajor_FLOW/=rmajor) pause 'WARNING: input inconsistent for R_major!'


	FV = b_phi_zero * rmajor
	FC = FV * Fc_o_Fv
	dedge = dcenter * de_o_dc
	pcenter = beta_center*b_phi_zero*b_phi_zero/2.0d0/rmu0
	pedge = pcenter * pe_o_pc

	if(numerical_input_option>0) then
		numerical_p = .false.
		numerical_Bz = .false.
		numerical_n = .false.
	elseif(numerical_input_option<0) then
		numerical_p = .true.
		numerical_Bz = .true.
		numerical_n = .true.
		call read_numerical
	endif

	open (17, file='FLOW_n.dat', status='old', action='read')
	read(17,*) n_FLOW !this still assumes same number of points in each direction
	close(17)

	allocate(x_FLOW(n_FLOW), y_FLOW(n_FLOW))

	allocate(psi_FLOW(n_FLOW,n_FLOW))
	allocate(Bx_FLOW(n_FLOW,n_FLOW))
	allocate(By_FLOW(n_FLOW,n_FLOW))

	call read_FLOW_data(psi_FLOW,n_FLOW,n_FLOW,'psi')

	psic = maxval(psi_FLOW)

	! read grid

    open (unit=619,file='FLOW_xgrid.dat')

	do i=1,n_FLOW
		read(619,88) x_FLOW(i)
	end do

	close(619)

    open (unit=619,file='FLOW_zgrid.dat')

	do j=1,n_FLOW
		read(619,88) y_FLOW(j)
	end do

	close(619)

	dx2_FLOW = (x_FLOW(2)-x_FLOW(1))*2.d0
	dy2_FLOW = (y_FLOW(2)-y_FLOW(1))*2.d0

	! calculate poloidal field on FLOW grid
	Bx_FLOW = 0.d0; By_FLOW = 0.d0

	do j = 2, n_FLOW-1
	do i = 2, n_FLOW-1

		Rloc = x_FLOW(i)

		Bx_FLOW(i,j) = -(psi_FLOW(i,j+1) - psi_FLOW(i,j-1))/dy2_FLOW/Rloc
		By_FLOW(i,j) = (psi_FLOW(i+1,j) - psi_FLOW(i-1,j))/dx2_FLOW/Rloc

	enddo
	enddo

	! set up interpolations


	allocate (psi_bscoef(1:n_FLOW,1:n_FLOW))
	allocate (Bx_bscoef(1:n_FLOW,1:n_FLOW))
	allocate (By_bscoef(1:n_FLOW,1:n_FLOW))

	allocate (xknot(1:n_FLOW+FLOW_ord))
	allocate (yknot(1:n_FLOW+FLOW_ord))

	call DBSNAK(n_FLOW,x_FLOW,FLOW_ord,xknot)
	call DBSNAK(n_FLOW,y_FLOW,FLOW_ord,yknot)
	! (these 2 define the nodes)

	! set psi and B_poloidal

	call DBS2IN(n_FLOW,x_FLOW,n_FLOW,y_FLOW,psi_FLOW,n_FLOW,  &
				FLOW_ord,FLOW_ord,xknot,yknot,psi_bscoef(:,:) )
	call DBS2IN(n_FLOW,x_FLOW,n_FLOW,y_FLOW,Bx_FLOW,n_FLOW,  &
				FLOW_ord,FLOW_ord,xknot,yknot,Bx_bscoef(:,:) )
	call DBS2IN(n_FLOW,x_FLOW,n_FLOW,y_FLOW,By_FLOW,n_FLOW,  &
				FLOW_ord,FLOW_ord,xknot,yknot,By_bscoef(:,:) )

	! now fill in the initial values

	do j = 1, ny
	do i = 1, nx
!	do j = 0, ny+1
!	do i = 0, nx+1

		xloc = rmajor + xV(i) - xlength/2.d0
		yloc = yV(j) - ylength/2.d0

		if( (i==0).or.(i==nx+1).or.(j==0).or.(j==ny+1).or.  &
			(xloc<x_FLOW(1)).or.(xloc>x_FLOW(n_FLOW)).or.  &
			(yloc<y_FLOW(1)).or.(yloc>y_FLOW(n_FLOW)) ) then

			psiloc = 0.d0
			Bxloc = 0.d0
			Byloc = 0.d0

		else

			psiloc = DBS2VL(xloc,yloc,FLOW_ord,FLOW_ord,xknot, &
												yknot,n_FLOW,n_FLOW,psi_bscoef)

			Bxloc = DBS2VL(xloc,yloc,FLOW_ord,FLOW_ord,xknot, &
												yknot,n_FLOW,n_FLOW,Bx_bscoef)

			Byloc = DBS2VL(xloc,yloc,FLOW_ord,FLOW_ord,xknot, &
												yknot,n_FLOW,n_FLOW,By_bscoef)

			continue

		endif

		psi_init(i,j) = psiloc


		rho(i,j) = dofpsi(psiloc)
		p(i,j) = pofpsi(psiloc)

		! magnetic field
		bx(i,j) = Bxloc
		by(i,j) = Byloc
		bz(i,j) = Fofpsi(psiloc)/Rmaj(i)

		! velocities

		if((mach_theta_option==8).or.(mach_theta_option==9)) then
			phi_rho = mach_theta_max/sqrt(bx(i,j)**2+by(i,j)**2+1.d-32)
			if(psiloc/psic>2.d0*t_mth) then
				phi_rho=0.d0
			elseif(psiloc/psic>t_mth) then
				phi_rho = phi_rho*(1.d0-(psiloc/psic-t_mth)/t_mth)
			endif
		elseif((mach_theta_option==6).or.(mach_theta_option==7)) then
			phi_rho = mach_theta_max/sqrt(bx(i,j)**2+by(i,j)**2+bz(i,j)**2)
		else
			phi_rho = phiofpsi(psiloc)/rho(i,j)/sqrt(rmu0)
		endif

		if(mach_theta_option==7) then
			u(i,j) = phi_rho*bx(i,j)**2
			v(i,j) = phi_rho*by(i,j)**2
			w(i,j) = 0.d0
		else
			u(i,j) = phi_rho*bx(i,j)
			v(i,j) = phi_rho*by(i,j)
			if(mach_theta_option==9) then
!				omloc = -rsh*p(i,j)*rho(i,j)*bz(i,j)/rmajor*phi_rho
				omloc = -bz(i,j)/rmajor*phi_rho
			else
				omloc = omegaofpsi(psiloc)
			endif
			if(mach_theta_option/=8) w(i,j) = phi_rho*bz(i,j) + Rmaj(i)*omloc
		endif

		if(mach_theta_option==8)  w(i,j) = 0.d0


	enddo
	enddo

	psi_axis = maxval(psi_init)

	rhomin = dofpsi(0.d0)*1.d-3
	rho_boundary = dofpsi(0.d0)
	p_boundary = pofpsi(0.d0)

	! boundary

	! RIGHT
	bx(nx+1,:) = 0.d0; u(nx+1,:) = 0.d0

	! LEFT
	bx(0,:) = 0.d0; u(0,:) = 0.d0

	! TOP
	by(:,ny+1) = 0.d0; v(:,ny+1) = 0.d0

	! BOTTOM
	by(:,0) = 0.d0; v(:,0) = 0.d0

	! corners

	rho(0,0) = rho(1,1); rho(0,ny+1) = rho(1,ny); rho(nx+1,ny+1) = rho(nx,ny); rho(nx+1,0) = rho(nx,1)
	p(0,0) = p(1,1); p(0,ny+1) = p(1,ny); p(nx+1,ny+1) = p(nx,ny); p(nx+1,0) = p(nx,1)
	bx(0,0) = bx(1,1); bx(0,ny+1) = bx(1,ny); bx(nx+1,ny+1) = bx(nx,ny); bx(nx+1,0) = bx(nx,1)
	by(0,0) = by(1,1); by(0,ny+1) = by(1,ny); by(nx+1,ny+1) = by(nx,ny); by(nx+1,0) = by(nx,1)
	bz(0,0) = bz(1,1); bz(0,ny+1) = bz(1,ny); bz(nx+1,ny+1) = bz(nx,ny); bz(nx+1,0) = bz(nx,1)
	u(0,0) = u(1,1); u(0,ny+1) = u(1,ny); u(nx+1,ny+1) = u(nx,ny); u(nx+1,0) = u(nx,1)
	v(0,0) = v(1,1); v(0,ny+1) = v(1,ny); v(nx+1,ny+1) = v(nx,ny); v(nx+1,0) = v(nx,1)
	w(0,0) = w(1,1); w(0,ny+1) = w(1,ny); w(nx+1,ny+1) = w(nx,ny); w(nx+1,0) = w(nx,1)

	! deallocation section

	deallocate(x_FLOW, y_FLOW)

	deallocate(psi_FLOW, Bx_FLOW, By_FLOW)

	deallocate (psi_bscoef)
	deallocate (Bx_bscoef)
	deallocate (By_bscoef)


88   format(e26.17,3x)

	continue

	return


end subroutine read_GS_FLOW


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine read_GS_FLOW_all_B
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! this routine reads FLOW output and interpolates to create the input
! note the difference in notation y <-> z

	use mod_parameters, only : rsh, nx, ny, rhomin, xlength, ylength,  &
												numerical_input_option, rho_boundary, p_boundary
	use mod_arrays, only : rho, bx, by, bz, u, v, w, p, Rmaj, xV, yV
	use pseudo_IMSL, only :DBSNAK, DBS2IN, DBS2VL

	integer :: i, j
	character :: dummy_char
	real(kind=dkind) :: dummy
	integer :: n_FLOW
	real(kind=dkind), dimension(:,:), allocatable :: psi_FLOW, Bx_FLOW, By_FLOW, Bphi_FLOW
	real(kind=dkind), dimension(:,:), allocatable :: rho_FLOW, vx_FLOW, vy_FLOW, vphi_FLOW, p_FLOW
	real(kind=dkind), dimension(:), allocatable :: x_FLOW, y_FLOW
	real(kind=dkind) :: dx2_FLOW, dy2_FLOW, Rloc
	real(kind=dkind) :: psiloc, xloc, yloc, Bxloc, Byloc, Bphiloc, phi_rho
	real(kind=dkind), dimension(:), allocatable :: xknot, yknot
	real(kind=dkind), dimension(:,:), allocatable :: psi_bscoef, Bx_bscoef, By_bscoef, Bphi_bscoef,  &
																rho_bscoef, vx_bscoef, vy_bscoef, vphi_bscoef, p_bscoef
	integer :: FLOW_ord = 2	! interpolation order

	! initializations




	if(rmajor_FLOW/=rmajor) pause 'WARNING: input inconsistent for R_major!'


	FV = b_phi_zero * rmajor
	FC = FV * Fc_o_Fv
	dedge = dcenter * de_o_dc
	pcenter = beta_center*b_phi_zero*b_phi_zero/2.0d0/rmu0
	pedge = pcenter * pe_o_pc


	numerical_p = .false.
	numerical_Bz = .false.

	open (17, file='FLOW_n.dat', status='old', action='read')
	read(17,*) n_FLOW !this still assumes same number of points in each direction
	close(17)

	allocate(x_FLOW(n_FLOW), y_FLOW(n_FLOW))

	allocate(psi_FLOW(n_FLOW,n_FLOW))
	allocate(Bx_FLOW(n_FLOW,n_FLOW))
	allocate(By_FLOW(n_FLOW,n_FLOW))
	allocate(Bphi_FLOW(n_FLOW,n_FLOW))

	call read_FLOW_data(psi_FLOW,n_FLOW,n_FLOW,'psi')

	call read_FLOW_data(Bx_FLOW,n_FLOW,n_FLOW,'br')
	call read_FLOW_data(By_FLOW,n_FLOW,n_FLOW,'bz')
	call read_FLOW_data(Bphi_FLOW,n_FLOW,n_FLOW,'b_phi')

	! if other quantities are required, read those, too

	if(numerical_input_option==9) then

		allocate(rho_FLOW(n_FLOW,n_FLOW))
		allocate(vx_FLOW(n_FLOW,n_FLOW))
		allocate(vy_FLOW(n_FLOW,n_FLOW))
		allocate(vphi_FLOW(n_FLOW,n_FLOW))
		allocate(p_FLOW(n_FLOW,n_FLOW))

		call read_FLOW_data(rho_FLOW,n_FLOW,n_FLOW,'rho')
		call read_FLOW_data(vx_FLOW,n_FLOW,n_FLOW,'vr')
		call read_FLOW_data(vy_FLOW,n_FLOW,n_FLOW,'vz')
		call read_FLOW_data(vphi_FLOW,n_FLOW,n_FLOW,'vphi')
		call read_FLOW_data(p_FLOW,n_FLOW,n_FLOW,'pres')

	endif

	psic = maxval(psi_FLOW)

	! read grid

    open (unit=619,file='FLOW_xgrid.dat')

	do i=1,n_FLOW
		read(619,88) x_FLOW(i)
	end do

	close(619)

    open (unit=619,file='FLOW_zgrid.dat')

	do j=1,n_FLOW
		read(619,88) y_FLOW(j)
	end do

	close(619)

	dx2_FLOW = (x_FLOW(2)-x_FLOW(1))*2.d0
	dy2_FLOW = (y_FLOW(2)-y_FLOW(1))*2.d0

	! set up interpolations


	allocate (psi_bscoef(1:n_FLOW,1:n_FLOW))
	allocate (Bx_bscoef(1:n_FLOW,1:n_FLOW))
	allocate (By_bscoef(1:n_FLOW,1:n_FLOW))
	allocate (Bphi_bscoef(1:n_FLOW,1:n_FLOW))

	if(numerical_input_option==9) then

		allocate (rho_bscoef(1:n_FLOW,1:n_FLOW))
		allocate (vx_bscoef(1:n_FLOW,1:n_FLOW))
		allocate (vy_bscoef(1:n_FLOW,1:n_FLOW))
		allocate (vphi_bscoef(1:n_FLOW,1:n_FLOW))
		allocate (p_bscoef(1:n_FLOW,1:n_FLOW))

	endif


	allocate (xknot(1:n_FLOW+FLOW_ord))
	allocate (yknot(1:n_FLOW+FLOW_ord))

	call DBSNAK(n_FLOW,x_FLOW,FLOW_ord,xknot)
	call DBSNAK(n_FLOW,y_FLOW,FLOW_ord,yknot)
	! (these 2 define the nodes)

	! set psi and B

	call DBS2IN(n_FLOW,x_FLOW,n_FLOW,y_FLOW,psi_FLOW,n_FLOW,  &
				FLOW_ord,FLOW_ord,xknot,yknot,psi_bscoef(:,:) )
	call DBS2IN(n_FLOW,x_FLOW,n_FLOW,y_FLOW,Bx_FLOW,n_FLOW,  &
				FLOW_ord,FLOW_ord,xknot,yknot,Bx_bscoef(:,:) )
	call DBS2IN(n_FLOW,x_FLOW,n_FLOW,y_FLOW,By_FLOW,n_FLOW,  &
				FLOW_ord,FLOW_ord,xknot,yknot,By_bscoef(:,:) )
	call DBS2IN(n_FLOW,x_FLOW,n_FLOW,y_FLOW,Bphi_FLOW,n_FLOW,  &
				FLOW_ord,FLOW_ord,xknot,yknot,Bphi_bscoef(:,:) )

	if(numerical_input_option==9) then

		call DBS2IN(n_FLOW,x_FLOW,n_FLOW,y_FLOW,rho_FLOW,n_FLOW,  &
					FLOW_ord,FLOW_ord,xknot,yknot,rho_bscoef(:,:) )
		call DBS2IN(n_FLOW,x_FLOW,n_FLOW,y_FLOW,vx_FLOW,n_FLOW,  &
					FLOW_ord,FLOW_ord,xknot,yknot,vx_bscoef(:,:) )
		call DBS2IN(n_FLOW,x_FLOW,n_FLOW,y_FLOW,vy_FLOW,n_FLOW,  &
					FLOW_ord,FLOW_ord,xknot,yknot,vy_bscoef(:,:) )
		call DBS2IN(n_FLOW,x_FLOW,n_FLOW,y_FLOW,vphi_FLOW,n_FLOW,  &
					FLOW_ord,FLOW_ord,xknot,yknot,vphi_bscoef(:,:) )
		call DBS2IN(n_FLOW,x_FLOW,n_FLOW,y_FLOW,p_FLOW,n_FLOW,  &
					FLOW_ord,FLOW_ord,xknot,yknot,p_bscoef(:,:) )

	endif

	! now fill in the initial values

	do j = 1, ny
	do i = 1, nx

		xloc = rmajor + xV(i) - xlength/2.d0
		yloc = yV(j) - ylength/2.d0

		if( (i==0).or.(i==nx+1).or.(j==0).or.(j==ny+1) ) then

			psiloc = 0.d0

		else

			psiloc = DBS2VL(xloc,yloc,FLOW_ord,FLOW_ord,xknot, &
												yknot,n_FLOW,n_FLOW,psi_bscoef)

		endif

		psi_init(i,j) = psiloc

		Bxloc = DBS2VL(xloc,yloc,FLOW_ord,FLOW_ord,xknot, &
											yknot,n_FLOW,n_FLOW,Bx_bscoef)

		Byloc = DBS2VL(xloc,yloc,FLOW_ord,FLOW_ord,xknot, &
											yknot,n_FLOW,n_FLOW,By_bscoef)

		Bphiloc = DBS2VL(xloc,yloc,FLOW_ord,FLOW_ord,xknot, &
											yknot,n_FLOW,n_FLOW,Bphi_bscoef)

		if(numerical_input_option==9) then

			rho(i,j) = DBS2VL(xloc,yloc,FLOW_ord,FLOW_ord,xknot, &
											yknot,n_FLOW,n_FLOW,rho_bscoef)
			p(i,j) = DBS2VL(xloc,yloc,FLOW_ord,FLOW_ord,xknot, &
											yknot,n_FLOW,n_FLOW,p_bscoef)

			! velocities

			u(i,j) = DBS2VL(xloc,yloc,FLOW_ord,FLOW_ord,xknot, &
											yknot,n_FLOW,n_FLOW,vx_bscoef)
			v(i,j) = DBS2VL(xloc,yloc,FLOW_ord,FLOW_ord,xknot, &
											yknot,n_FLOW,n_FLOW,vy_bscoef)
			w(i,j) = DBS2VL(xloc,yloc,FLOW_ord,FLOW_ord,xknot, &
											yknot,n_FLOW,n_FLOW,vphi_bscoef)

		else

			rho(i,j) = dofpsi(psiloc)
			p(i,j) = pofpsi(psiloc)

			! velocities
			if((mach_theta_option==8).or.(mach_theta_option==9)) then
				phi_rho = mach_theta_max/sqrt(bx(i,j)**2+by(i,j)**2+1.d-32)
				if(psiloc/psic>2.d0*t_mth) then
					phi_rho=0.d0
				elseif(psiloc/psic>t_mth) then
					phi_rho = phi_rho*(1.d0-(psiloc/psic-t_mth)/t_mth)
				endif
			elseif((mach_theta_option==6).or.(mach_theta_option==7)) then
				phi_rho = mach_theta_max/sqrt(bx(i,j)**2+by(i,j)**2+bz(i,j)**2)
			else
				phi_rho = phiofpsi(psiloc)/rho(i,j)/sqrt(rmu0)
			endif

			if(mach_theta_option==7) then
				u(i,j) = phi_rho*bx(i,j)**2
				v(i,j) = phi_rho*by(i,j)**2
				w(i,j) = 0.d0
			else
				u(i,j) = phi_rho*bx(i,j)
				v(i,j) = phi_rho*by(i,j)
				w(i,j) = phi_rho*bz(i,j) + Rmaj(i)*omegaofpsi(psiloc)
			endif

			if(mach_theta_option==8)  w(i,j) = 0.d0


		endif

		! magnetic field
		bx(i,j) = Bxloc
		by(i,j) = Byloc
		bz(i,j) = Bphiloc	!Fofpsi(psiloc)/Rmaj(i)


	enddo
	enddo

	psi_axis = maxval(psi_init)

	call read_numerical

	rhomin = dofpsi(0.d0)*1.d-3
	rho_boundary = dofpsi(0.d0)

	p_boundary = pofpsi(0.d0)

	! boundary

	! RIGHT
	bx(nx+1,:) = 0.d0; u(nx+1,:) = 0.d0

	! LEFT
	bx(0,:) = 0.d0; u(0,:) = 0.d0

	! TOP
	by(:,ny+1) = 0.d0; v(:,ny+1) = 0.d0

	! BOTTOM
	by(:,0) = 0.d0; v(:,0) = 0.d0

	! corners

	if(numerical_input_option<9) then
		rho(0,0) = rho(1,1); rho(0,ny+1) = rho(1,ny); rho(nx+1,ny+1) = rho(nx,ny); rho(nx+1,0) = rho(nx,1)
		p(0,0) = p(1,1); p(0,ny+1) = p(1,ny); p(nx+1,ny+1) = p(nx,ny); p(nx+1,0) = p(nx,1)
		u(0,0) = u(1,1); u(0,ny+1) = u(1,ny); u(nx+1,ny+1) = u(nx,ny); u(nx+1,0) = u(nx,1)
		v(0,0) = v(1,1); v(0,ny+1) = v(1,ny); v(nx+1,ny+1) = v(nx,ny); v(nx+1,0) = v(nx,1)
		w(0,0) = w(1,1); w(0,ny+1) = w(1,ny); w(nx+1,ny+1) = w(nx,ny); w(nx+1,0) = w(nx,1)
	endif

!	bx(0,0) = bx(1,1); bx(0,ny+1) = bx(1,ny); bx(nx+1,ny+1) = bx(nx,ny); bx(nx+1,0) = bx(nx,1)
!	by(0,0) = by(1,1); by(0,ny+1) = by(1,ny); by(nx+1,ny+1) = by(nx,ny); by(nx+1,0) = by(nx,1)
!	bz(0,0) = bz(1,1); bz(0,ny+1) = bz(1,ny); bz(nx+1,ny+1) = bz(nx,ny); bz(nx+1,0) = bz(nx,1)

	! deallocation section

	deallocate(x_FLOW, y_FLOW)

	deallocate(psi_FLOW, Bx_FLOW, By_FLOW, Bphi_FLOW)

	deallocate (psi_bscoef)
	deallocate (Bx_bscoef)
	deallocate (By_bscoef)
	deallocate (Bphi_bscoef)

	if(numerical_input_option==9) then

		deallocate(rho_FLOW, vx_FLOW, vy_FLOW, vphi_FLOW,p_FLOW)

		deallocate (rho_bscoef,vx_bscoef,vy_bscoef,vphi_bscoef,p_bscoef)

	endif

88   format(e26.17,3x)

	continue

	return


end subroutine read_GS_FLOW_all_B

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  
subroutine read_FLOW_data(thing,nx,nz,fname)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  

	implicit none

	integer, intent(in) :: nx,nz
	real (kind=dkind), dimension(1:nx,1:nz) :: thing
	character (len=*), intent(in) :: fname
	integer i,j

	open (unit=17, file=fname//'bin.out', form='unformatted', status='old', action='read')

	do 33 j=1,nz
	do 33 i=1,nx

	33 read(17) thing(i,j)

	close(17)

	continue

	return

end subroutine read_FLOW_data


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
function pofpsi(psiloc) result(answer)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	real(kind=dkind) :: psiloc, answer

	if(numerical_p) then

		if(psiloc<=0.d0) then

			answer = dbsval(0.d0, p_ord, p_data(1:ibreak_p,3),  &
			ibreak_p-p_ord, p_cscoef(1,1:ibreak_p-p_ord) )

		elseif(psiloc/psic>1.d0) then

			answer = dbsval(1.d0, p_ord, p_data(1:ibreak_p,3),  &
			ibreak_p-p_ord, p_cscoef(1,1:ibreak_p-p_ord) )

		else

			answer = dbsval(psiloc/psic, p_ord, p_data(1:ibreak_p,3),  &
				ibreak_p-p_ord, p_cscoef(1,1:ibreak_p-p_ord) )

		endif

	else

		if(psiloc<=0.d0) then

			answer = pedge

		elseif(psiloc/psic>1.d0) then

			answer = pcenter

		else

			answer = pedge + (pcenter-pedge)* (psiloc/psic)**alpha

		endif

	endif

	answer = answer + p_add

	return

end function pofpsi


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
function Fofpsi(psiloc) result(answer)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	real(kind=dkind) :: psiloc, answer, Bzloc

	if(numerical_Bz) then

		if(psiloc<=0.d0) then

			Bzloc = dbsval(0.d0, Bz_ord, Bz_data(1:ibreak_Bz,3),  &
					ibreak_Bz-Bz_ord, Bz_cscoef(1,1:ibreak_Bz-Bz_ord) )

		elseif(psiloc/psic>1.d0) then

			Bzloc = dbsval(1.d0, Bz_ord, Bz_data(1:ibreak_Bz,3),  &
					ibreak_Bz-Bz_ord, Bz_cscoef(1,1:ibreak_Bz-Bz_ord) )

		else

			Bzloc = dbsval(psiloc/psic, Bz_ord, Bz_data(1:ibreak_Bz,3),  &
					ibreak_Bz-Bz_ord, Bz_cscoef(1,1:ibreak_Bz-Bz_ord) )

		endif

		

		answer = rmajor * Bzloc

	else

		answer = FV + (FC - FV)*(psiloc/psic)**kappa

	endif

	return

end function Fofpsi


!-------------------------------functions not needed for the equilibrium-------------------------------
! pofpsi is listed before, even though it's not needed for the equilibrium just to keep it close to dpdpsi

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
function dofpsi(psiloc) result(answer)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	real(kind=dkind) :: psiloc, answer


	if(numerical_n) then

		if(psiloc<=0.d0) then

			answer = dbsval(0.d0, d_ord, d_data(1:ibreak_d,3),  &
				ibreak_d-d_ord, d_cscoef(1,1:ibreak_d-d_ord) )

		elseif(psiloc/psic>1.d0) then

			answer = dbsval(1.d0, d_ord, d_data(1:ibreak_d,3),  &
				ibreak_d-d_ord, d_cscoef(1,1:ibreak_d-d_ord) )

		else

			answer = dbsval(psiloc/psic, d_ord, d_data(1:ibreak_d,3),  &
				ibreak_d-d_ord, d_cscoef(1,1:ibreak_d-d_ord) )
		endif

	else

		if(psiloc<=0.d0) then

			answer = dedge

		elseif(psiloc/psic>1.d0) then

			answer = dcenter

		else

			answer = dedge + (dcenter-dedge)* (psiloc/psic)**alpha_rho

		endif

	endif

	return

end function dofpsi

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  
  function mach_theta(psiloc) result (answer)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  

    real (kind=dkind), intent (in) :: psiloc
    real (kind=dkind) :: answer

	if(mach_theta_option==1) then

		!------------------ normal shape!----------------------
		if(psiloc <= 0.d0) then
			answer = mach_theta_edge
		else
			answer = mach_theta_edge + (mach_theta_max-mach_theta_edge)*  &
							4.0d0*(psiloc/psic)*(1.0d0 - psiloc/psic)  
		end if

	elseif(mach_theta_option==2) then

		!------------------ linear shape for transonic flow----------------------
		if(psiloc <= 0.d0) then
			answer = mach_theta_max
		else
			answer = mach_theta_max*(1.d0-psiloc/psic)
		end if

	elseif(mach_theta_option==3) then

		!------------------ power-law shape for transonic flow----------------------
		if(psiloc <= 0.d0) then
			answer = mach_theta_max
		else
			answer = mach_theta_max*(1.d0-psiloc/psic)**3
		end if

	elseif(mach_theta_option==4) then

			! -------------------- t-2t shape --------------------

		if(psiloc<=0.d0) then
			answer = mach_theta_edge
		elseif(psiloc/psic>=2.d0*t_mth) then
			answer = 0.d0
		elseif(psiloc/psic<=t_mth) then
			answer = mach_theta_edge + (mach_theta_max-mach_theta_edge) *  &
					(2.d0/t_mth*psiloc/psic - (psiloc/psic/t_mth)**2)
		elseif(psiloc/psic>t_mth) then
			answer = mach_theta_max *  &
					(2.d0*t_mth-psiloc/psic)**2 * (2.d0*psiloc/psic-t_mth) / t_mth**3
		endif

	elseif(mach_theta_option==5) then

			! -------------------- t-straight shape --------------------

		if(psiloc<=0.d0) then
			answer = mach_theta_edge
		elseif(psiloc/psic>=t_mth) then
			answer = 0.d0
		elseif(psiloc/psic<=t_mth) then
			answer = mach_theta_edge * (1.d0 - (psiloc/psic/t_mth))

		endif

	endif

	continue

  end function mach_theta


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  
  function mach_phi(psiloc) result (answer)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  

    real (kind=dkind), intent (in) :: psiloc
    real (kind=dkind) :: answer

	answer = mphi_min + (mach_phi_max-mphi_min)*  &
						(psiloc/psic)**alpha_mphi


	continue

  end function mach_phi

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  
  function phiofpsi(psiloc) result(answer)		
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  

	use mod_parameters, only : rsh

    real (kind=dkind), intent (in) :: psiloc
    real (kind=dkind) :: answer
	real (kind=dkind) :: cs, b_phi_zero

	answer = dsqrt(rmu0*rsh*pofpsi(psiloc)*dofpsi(psiloc))* &
					 mach_theta(psiloc)*rmajor/Fofpsi(psiloc)


    return

  end function phiofpsi


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  
  function omegaofpsi(psiloc) result(answer)		
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  

	use mod_parameters, only : rsh

    real (kind=dkind), intent (in) :: psiloc
    real (kind=dkind) :: answer
	real (kind=dkind) :: cs

	answer = dsqrt(rsh*pofpsi(psiloc)/dofpsi(psiloc))* &
						(mach_phi(psiloc) - mach_theta(psiloc))/rmajor

    return

  end function omegaofpsi





end module GS_equilibrium
