module boundary_routines

	use mod_parameters, only : dkind, rho_boundary, p_boundary

	implicit none

	private

	public  sort_grid, boundary_cleanup,  &
								 boundary_reflect_square_two, set_boundary_square, boundary_reflect_square,  &
								 boundary_reflect_square_vortex_model, set_current, boundary_reflect_square_current,  &
								 boundary_reflect_square_rhoflat, boundary_vacuum_0, boundary_vacuum_1,  &
								 boundary_vacuum_field_copy, Mach_boundary, boundary_vacuum_vn0,  &
								 boundary_vacuum_CS, boundary_vacuum_CS_2, boundary_vacuum_CS_3, set_boundary_vacuum,  &
								 set_boundary_values


	integer, dimension(:,:) , allocatable :: sort_grid

	real(kind=dkind) :: Mach_boundary = 0.1d0

	integer, parameter :: SW=1, SE=2, NE=3, NW=4

	contains


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine set_boundary_values
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	real(kind=dkind) :: dummy

	open(33, file='./input/n.dat',status='old',action='read')
	read(33,*) dummy, rho_boundary
	close(33)

	open(33, file='./input/p.dat',status='old',action='read')
	read(33,*) dummy, p_boundary
	close(33)

	continue

	return

end subroutine set_boundary_values

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine set_boundary_square(nx,ny)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! note: this assumes the domain to be square
! very little needs to be done here

	use mod_parameters, only : dx, dy, dA !, vacuum_region

	integer :: nx, ny

	allocate(sort_grid(0:nx+1,0:ny+1))
	sort_grid = 0
	sort_grid(1:nx,1:ny) = 1
	! this is needed only for compatibility with routines using sort_grid

	dA = dx*dy


end subroutine set_boundary_square

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine set_boundary_vacuum
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! note: this assumes the domain to be square
! very little needs to be done here

	use mod_parameters, only : nx, ny, dx, dy, dA, external_surface
	use vacuum_module, only : radius

	real(kind=dkind) :: ex, ey, rloc
	integer :: i, j

	allocate(sort_grid(0:nx+1,0:ny+1))
	sort_grid = 0
	sort_grid(1:nx,1:ny) = 1
	! this is needed only for compatibility with routines using sort_grid

	dA = dx*dy

	external_surface = 0.d0

	do j = 1, ny
	do i = 1, nx

		call radius(i,j,ex,ey,rloc)

		if(ex**2+ey**2>rloc) then
		! external point

			external_surface = external_surface + dA

		else
		! internal point

			sort_grid(i,j) = 2

		endif

	enddo
	enddo


end subroutine set_boundary_vacuum

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine set_current
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! calculate total plasma current to be used in magnetic field boundary condition

	use mod_parameters, only : nx, ny, dx, dy, current
	use mod_arrays, only : bx, by

	implicit none

	real(kind=dkind) :: dA, Jz, current2
	integer :: i, j

	dA = dx*dy

	current = 0.d0
	current2 = 0.d0

	do j = 2, ny-1
	do i = 2, nx-1

		Jz = (by(i+1,j)-by(i-1,j))/(2.d0*dx) -  &
			   (bx(i,j+1)-bx(i,j-1))/(2.d0*dy)

		current = current + Jz*dA

	enddo
	enddo

	do j = 2, ny

		current2 = current2 - by(1,j)*dy
		current2 = current2 + by(nx,j)*dy

	enddo

	do i = 2, nx

		current2 = current2 + bx(i,1)*dx
		current2 = current2 - bx(i,ny)*dx

	enddo

	continue

	return

end subroutine set_current


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine  boundary_reflect_square(nx,ny,nmx,nmy,u,v,w,rho,p,bx,by,bz)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	use mod_parameters, ONLY: dx, rsh, rmu0, torus, pmin
	use mod_arrays, only : Rmaj

	integer :: nmx, nmy, nx, ny
	integer :: i, j

	real(kind=dkind) ::  rho(0:nmx+1,0:nmy+1),u(0:nmx+1,0:nmy+1),  &
      v(0:nmx+1,0:nmy+1),w(0:nmx+1,0:nmy+1),  &
      p(0:nmx+1,0:nmy+1),  &
      bx(0:nmx+1,0:nmy+1),by(0:nmx+1,0:nmy+1),bz(0:nmx+1,0:nmy+1)


	do j=1,ny

		!!$   boundary conditions on left boundary (reflective)
		rho(0,j)=rho(1,j)
		u(0,j)=-u(1,j) !*rmaj(1)/rmaj(0)
		v(0,j)=v(1,j)    !*rmaj(1)/rmaj(0)
		w(0,j)=w(1,j)     *(rmaj(1)/rmaj(0))**2
		bx(0,j)=-bx(1,j)
		by(0,j)=by(1,j)    !*rmaj(1)/rmaj(0)
		bz(0,j)=bz(1,j)*rmaj(1)/rmaj(0)

		p(0,j)=p(1,j)+(by(1,j)**2-by(0,j)**2)/  &
			(2.d0*rmu0)-torus*((rho(1,j)*(w(1,j)**2+w(0,j)**2)  &
			)*dx/(rmaj(1)+rmaj(0)))
		p(0,j) = max(pmin,p(0,j))

		!!$   boundary conditions on right boundary (reflective)
		rho(nx+1,j)=rho(nx,j)
		u(nx+1,j)=-u(nx,j) ! *rmaj(nx)/rmaj(nx+1)
		v(nx+1,j)=v(nx,j)       !*rmaj(nx)/rmaj(nx+1)
		w(nx+1,j)=w(nx,j)       *(rmaj(nx)/rmaj(nx+1))**2
		bx(nx+1,j)=-bx(nx,j)
		by(nx+1,j)=by(nx,j)     !*rmaj(nx)/rmaj(nx+1)
		bz(nx+1,j)=bz(nx,j)*rmaj(nx)/rmaj(nx+1)



		p(nx+1,j)=p(nx,j)+(by(nx,j)**2-  &
		    by(nx+1,j)**2)/(2.*rmu0)+torus*(  &
		   (rho(nx,j)*(w(nx,j)**2+w(nx+1,j)**2)  &
		    )*dx/(rmaj(nx)+rmaj(nx+1)))
		p(nx+1,j) = max(pmin,p(nx+1,j))

	enddo




	do i=1,nx

		!!$     boundary conditions on the bottom boundary (reflective)
		rho(i,0)=rho(i,1)
		u(i,0)=u(i,1)
		v(i,0)=-v(i,1)
		w(i,0)=w(i,1)
		bx(i,0)=bx(i,1)
		by(i,0)=-by(i,1)
		bz(i,0)=bz(i,1)
		p(i,0)=p(i,1)

		!!$     boundary conditions on the top boundary (reflective)
		rho(i,ny+1)=rho(i,ny)
		u(i,ny+1)=u(i,ny)
		v(i,ny+1)=-v(i,ny)
		w(i,ny+1)=w(i,ny)
		bx(i,ny+1)=bx(i,ny)
		by(i,ny+1)=-by(i,ny)
		bz(i,ny+1)=bz(i,ny)
		p(i,ny+1)=p(i,ny)

	enddo

	return

end subroutine boundary_reflect_square

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine  boundary_reflect_square_rhoflat(nx,ny,nmx,nmy,u,v,w,rho,p,bx,by,bz)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	use mod_parameters, ONLY: dx, rsh, rmu0, torus, pmin
	use mod_arrays, only : Rmaj

	integer :: nmx, nmy, nx, ny
	integer :: i, j

	real(kind=dkind) ::  rho(0:nmx+1,0:nmy+1),u(0:nmx+1,0:nmy+1),  &
      v(0:nmx+1,0:nmy+1),w(0:nmx+1,0:nmy+1),  &
      p(0:nmx+1,0:nmy+1),  &
      bx(0:nmx+1,0:nmy+1),by(0:nmx+1,0:nmy+1),bz(0:nmx+1,0:nmy+1)


	do j=1,ny

		!!$   boundary conditions on left boundary (reflective)
		rho(0,j) = min(rho_boundary,rho(1,j))
		u(0,j)=-u(1,j)
		v(0,j)=v(1,j)    !*rmaj(1)/rmaj(0)
		w(0,j)=w(1,j)     *(rmaj(1)/rmaj(0))**2
		bx(0,j)=-bx(1,j)
		by(0,j)=by(1,j)    !*rmaj(1)/rmaj(0)
		bz(0,j)=bz(1,j)*rmaj(1)/rmaj(0)

		p(0,j)=p(1,j)+(by(1,j)**2-by(0,j)**2)/  &
			(2.*rmu0)-torus*((rho(1,j)*(w(1,j)**2+w(0,j)**2)  &
			)*dx/(rmaj(1)+rmaj(0)))
		p(0,j) = max(pmin,p(0,j))

		!!$   boundary conditions on right boundary (reflective)
		rho(nx+1,j) = min(rho(nx,j),rho_boundary)
		u(nx+1,j)=-u(nx,j)
		v(nx+1,j)=v(nx,j)       !*rmaj(nx)/rmaj(nx+1)
		w(nx+1,j)=w(nx,j)       *(rmaj(nx)/rmaj(nx+1))**2
		bx(nx+1,j)=-bx(nx,j)
		by(nx+1,j)=by(nx,j)     !*rmaj(nx)/rmaj(nx+1)
		bz(nx+1,j)=bz(nx,j)*rmaj(nx)/rmaj(nx+1)

		p(nx+1,j)=p(nx,j)+(by(nx,j)**2-  &
		    by(nx+1,j)**2)/(2.*rmu0)+torus*(  &
		   (rho(nx,j)*(w(nx,j)**2+w(nx+1,j)**2)  &
		    )*dx/(rmaj(nx)+rmaj(nx+1)))
		p(nx+1,j) = max(pmin,p(nx+1,j))

	enddo




	do i=1,nx

		!!$     boundary conditions on the bottom boundary (reflective)
		rho(i,0) = min(rho(i,1),rho_boundary)
		u(i,0)=u(i,1)
		v(i,0)=-v(i,1)
		w(i,0)=w(i,1)
		bx(i,0)=bx(i,1)
		by(i,0)=-by(i,1)
		bz(i,0)=bz(i,1)
		p(i,0)=p(i,1)

		!!$     boundary conditions on the top boundary (reflective)
		rho(i,ny+1) = min(rho_boundary,rho(i,ny))
		u(i,ny+1)=u(i,ny)
		v(i,ny+1)=-v(i,ny)
		w(i,ny+1)=w(i,ny)
		bx(i,ny+1)=bx(i,ny)
		by(i,ny+1)=-by(i,ny)
		bz(i,ny+1)=bz(i,ny)
		p(i,ny+1)=p(i,ny)

	enddo

	return

end subroutine boundary_reflect_square_rhoflat


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine  boundary_reflect_square_current(nx,ny,nmx,nmy,u,v,w,rho,p,bx,by,bz)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	use mod_parameters, ONLY: dx, dy, rsh, rmu0, torus, pmin, current
	use mod_arrays, only : Rmaj

	integer :: nmx, nmy, nx, ny
	integer :: i, j

	real(kind=dkind) ::  rho(0:nmx+1,0:nmy+1),u(0:nmx+1,0:nmy+1),  &
      v(0:nmx+1,0:nmy+1),w(0:nmx+1,0:nmy+1),  &
      p(0:nmx+1,0:nmy+1),  &
      bx(0:nmx+1,0:nmy+1),by(0:nmx+1,0:nmy+1),bz(0:nmx+1,0:nmy+1)

	real(kind=dkind) :: current_current, current_factor




	do j=1,ny

		!!$   boundary conditions on left boundary (reflective)
		rho(0,j)=rho(1,j)
		u(0,j)=-u(1,j)
		v(0,j)=v(1,j)    !*rmaj(1)/rmaj(0)
		w(0,j)=w(1,j)     *(rmaj(1)/rmaj(0))**2
		bx(0,j)=-bx(1,j)
		by(0,j)=by(1,j)    !*rmaj(1)/rmaj(0)
		bz(0,j)=bz(1,j)*rmaj(1)/rmaj(0)

		p(0,j)=p(1,j)+(by(1,j)**2-by(0,j)**2)/  &
			(2.*rmu0)-torus*((rho(1,j)*(w(1,j)**2+w(0,j)**2)  &
			)*dx/(rmaj(1)+rmaj(0)))
		p(0,j) = max(pmin,p(0,j))

		!!$   boundary conditions on right boundary (reflective)
		rho(nx+1,j)=rho(nx,j)
		u(nx+1,j)=-u(nx,j)
		v(nx+1,j)=v(nx,j)       !*rmaj(nx)/rmaj(nx+1)
		w(nx+1,j)=w(nx,j)       *(rmaj(nx)/rmaj(nx+1))**2
		bx(nx+1,j)=-bx(nx,j)
		by(nx+1,j)=by(nx,j)     !*rmaj(nx)/rmaj(nx+1)
		bz(nx+1,j)=bz(nx,j)*rmaj(nx)/rmaj(nx+1)



		p(nx+1,j)=p(nx,j)+(by(nx,j)**2-  &
		    by(nx+1,j)**2)/(2.*rmu0)+torus*(  &
		   (rho(nx,j)*(w(nx,j)**2+w(nx+1,j)**2)  &
		    )*dx/(rmaj(nx)+rmaj(nx+1)))
		p(nx+1,j) = max(pmin,p(nx+1,j))

	enddo




	do i=1,nx

		!!$     boundary conditions on the bottom boundary (reflective)
		rho(i,0)=rho(i,1)
		u(i,0)=u(i,1)
		v(i,0)=-v(i,1)
		w(i,0)=w(i,1)
		bx(i,0)=bx(i,1)
		by(i,0)=-by(i,1)
		bz(i,0)=bz(i,1)
		p(i,0)=p(i,1)

		!!$     boundary conditions on the top boundary (reflective)
		rho(i,ny+1)=rho(i,ny)
		u(i,ny+1)=u(i,ny)
		v(i,ny+1)=-v(i,ny)
		w(i,ny+1)=w(i,ny)
		bx(i,ny+1)=bx(i,ny)
		by(i,ny+1)=-by(i,ny)
		bz(i,ny+1)=bz(i,ny)
		p(i,ny+1)=p(i,ny)

	enddo

	! now determine the current current (pun intended)

	current_current = 0.d0

	do j = 2, ny

		current_current = current_current - by(0,j)*dy
		current_current = current_current + by(nx+1,j)*dy

	enddo

	do i = 2, nx

		current_current = current_current + bx(i,0)*dx
		current_current = current_current - bx(i,ny+1)*dx

	enddo

	current_factor = current / current_current

	do j = 2, ny

		by(0,j) = by(0,j)*current_factor
		by(nx+1,j) = by(nx+1,j)*current_factor

	enddo

	do i = 2, nx

		bx(i,0) = bx(i,0)*current_factor
		bx(i,ny+1) = bx(i,ny+1)*current_factor

	enddo

	return

end subroutine boundary_reflect_square_current


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
  subroutine  boundary_reflect_square_two(nx,ny,nmx,nmy,u,v,w,rho,p,bx,by,bz)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	use mod_parameters, ONLY: dx, dy, rsh, rmu0, torus, pi, pmin
	use mod_arrays, only : Rmaj

	integer :: nmx, nmy, nx, ny
	integer :: i, j

	real(kind=dkind) ::  rho(0:nmx+1,0:nmy+1),u(0:nmx+1,0:nmy+1),  &
      v(0:nmx+1,0:nmy+1),w(0:nmx+1,0:nmy+1),  &
      p(0:nmx+1,0:nmy+1),  &
      bx(0:nmx+1,0:nmy+1),by(0:nmx+1,0:nmy+1),bz(0:nmx+1,0:nmy+1)

	real(kind=dkind) :: vn, vt, bn, bt

	real(kind=dkind), dimension(1:4) :: A_Qbar
	real(kind=dkind) :: vol


	do j=1,ny

		!!$   boundary conditions on left boundary (reflective)
		rho(0,j)=rho(1,j)
		u(0,j)=-u(1,j)
		v(0,j)=v(1,j)    !*rmaj(1)/rmaj(0)
		w(0,j)=w(1,j)     *(rmaj(1)/rmaj(0))**2
		bx(0,j)=-bx(1,j)
		by(0,j)=by(1,j)    !*rmaj(1)/rmaj(0)
		bz(0,j)=bz(1,j)*rmaj(1)/rmaj(0)

		p(0,j)=p(1,j)+(by(1,j)**2-by(0,j)**2)/  &
			(2.*rmu0)-torus*((rho(1,j)*(w(1,j)**2+w(0,j)**2)  &
			)*dx/(rmaj(1)+rmaj(0)))


		!!$   boundary conditions on right boundary (reflective)
		rho(nx+1,j)=rho(nx,j)
		u(nx+1,j)=-u(nx,j)
		v(nx+1,j)=v(nx,j)       !*rmaj(nx)/rmaj(nx+1)
		w(nx+1,j)=w(nx,j)       *(rmaj(nx)/rmaj(nx+1))**2
		bx(nx+1,j)=-bx(nx,j)
		by(nx+1,j)=by(nx,j)     !*rmaj(nx)/rmaj(nx+1)
		bz(nx+1,j)=bz(nx,j)*rmaj(nx)/rmaj(nx+1)



		p(nx+1,j)=p(nx,j)+(by(nx,j)**2-  &
		    by(nx+1,j)**2)/(2.*rmu0)+torus*(  &
		   (rho(nx,j)*(w(nx,j)**2+w(nx+1,j)**2)  &
		    )*dx/(rmaj(nx)+rmaj(nx+1)))

	enddo




	do i=1,nx

		!!$     boundary conditions on the bottom boundary (reflective)
		rho(i,0)=rho(i,1)
		u(i,0)=u(i,1)
		v(i,0)=-v(i,1)
		w(i,0)=w(i,1)
		bx(i,0)=bx(i,1)
		by(i,0)=-by(i,1)
		bz(i,0)=bz(i,1)
		p(i,0)=p(i,1)

		!!$     boundary conditions on the top boundary (reflective)
		rho(i,ny+1)=rho(i,ny)
		u(i,ny+1)=u(i,ny)
		v(i,ny+1)=-v(i,ny)
		w(i,ny+1)=w(i,ny)
		bx(i,ny+1)=bx(i,ny)
		by(i,ny+1)=-by(i,ny)
		bz(i,ny+1)=bz(i,ny)
		p(i,ny+1)=p(i,ny)

	enddo

	!-------------------------------------------------------------
	!do something special in the corners:
	! reconstruct corner points from the inside
	!-------------------------------------------------------------

	! ----------------------(1,1)----------------------
	i = 2; j = 2

	vol = 8.d0*pi*dx*dy*Rmaj(i)

	A_Qbar(SW) = dx*dy*pi*(-dx + 2.d0*Rmaj(i))
	A_Qbar(SE) = dx*dy*Pi*(dx + 2.d0*Rmaj(i))
	A_Qbar(NE) = dx*dy*Pi*(dx + 2.d0*Rmaj(i))
	A_Qbar(NW) = dx*dy*Pi*(-dx + 2.d0*Rmaj(i))
			!-------------------------

	rho(1,1) = ( rho(i,j) * vol -  &
					  rho(i+1,j-1) * A_Qbar(SE) -  &
					  rho(i+1,j+1) * A_Qbar(NE) -  &
					  rho(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(SW)

	u(1,1) = ( u(i,j) * vol -  &
				  u(i+1,j-1) * A_Qbar(SE) -  &
				  u(i+1,j+1) * A_Qbar(NE) -  &
				  u(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(SW)

	v(i,j) = ( v(i,j) * vol -  &
				  v(i+1,j-1) * A_Qbar(SE) -  &
				  v(i+1,j+1) * A_Qbar(NE) -  &
				  v(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(SW)

	w(1,1) = ( w(i,j) * vol -  &
				   w(i+1,j-1) * A_Qbar(SE) -  &
				   w(i+1,j+1) * A_Qbar(NE) -  &
				   w(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(SW)

	p(1,1) = ( p(i,j) * vol -  &
				  p(i+1,j-1) * A_Qbar(SE) -  &
				  p(i+1,j+1) * A_Qbar(NE) -  &
				  p(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(SW)

	p(1,1) = max(p(1,1),pmin)

!!$	bx(1,1) = ( bx(i,j) * vol -  &
!!$					bx(i+1,j-1) * A_Qbar(SE) -  &
!!$					bx(i+1,j+1) * A_Qbar(NE) -  &
!!$					bx(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(SW)
!!$
!!$	by(1,1) = ( by(i,j) * vol -  &
!!$					by(i+1,j-1) * A_Qbar(SE) -  &
!!$					by(i+1,j+1) * A_Qbar(NE) -  &
!!$					by(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(SW)
!!$
!!$	bz(1,1) = ( bz(i,j) * vol -  &
!!$				   bz(i+1,j-1) * A_Qbar(SE) -  &
!!$				   bz(i+1,j+1) * A_Qbar(NE) -  &
!!$				   bz(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(SW)


	! ----------------------(nx,1)----------------------
	i = nx-1; j = 2

	vol = 8.d0*pi*dx*dy*Rmaj(i)

	A_Qbar(SW) = dx*dy*pi*(-dx + 2.d0*Rmaj(i))
	A_Qbar(SE) = dx*dy*Pi*(dx + 2.d0*Rmaj(i))
	A_Qbar(NE) = dx*dy*Pi*(dx + 2.d0*Rmaj(i))
	A_Qbar(NW) = dx*dy*Pi*(-dx + 2.d0*Rmaj(i))
			!-------------------------

	rho(nx,1) = ( rho(i,j) * vol -  &
					  rho(i-1,j-1) * A_Qbar(SW) -  &
					  rho(i+1,j+1) * A_Qbar(NE) -  &
					  rho(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(SE)

	u(nx,1) = ( u(i,j) * vol -  &
				  u(i-1,j-1) * A_Qbar(SW) -  &
				  u(i+1,j+1) * A_Qbar(NE) -  &
				  u(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(SE)

	v(nx,1) = ( v(i,j) * vol -  &
				  v(i-1,j-1) * A_Qbar(SW) -  &
				  v(i+1,j+1) * A_Qbar(NE) -  &
				  v(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(SE)

	w(nx,1) = ( w(i,j) * vol -  &
				   w(i-1,j-1) * A_Qbar(SW) -  &
				   w(i+1,j+1) * A_Qbar(NE) -  &
				   w(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(SE)

	p(nx,1) = ( p(i,j) * vol -  &
				  p(i-1,j-1) * A_Qbar(SW) -  &
				  p(i+1,j+1) * A_Qbar(NE) -  &
				  p(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(SE)
	p(nx,1) = max(p(nx,1),pmin)

!!$	bx(nx,1) = ( bx(i,j) * vol -  &
!!$					bx(i-1,j-1) * A_Qbar(SW) -  &
!!$					bx(i+1,j+1) * A_Qbar(NE) -  &
!!$					bx(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(SE)
!!$
!!$	by(nx,1) = ( by(i,j) * vol -  &
!!$					by(i-1,j-1) * A_Qbar(SW) -  &
!!$					by(i+1,j+1) * A_Qbar(NE) -  &
!!$					by(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(SE)
!!$
!!$	bz(nx,1) = ( bz(i,j) * vol -  &
!!$					bz(i-1,j-1) * A_Qbar(SW) -  &
!!$					bz(i+1,j+1) * A_Qbar(NE) -  &
!!$					bz(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(SE)


	! ----------------------(nx,ny)----------------------
	i = nx-1; j = ny-1

	vol = 8.d0*pi*dx*dy*Rmaj(i)

	A_Qbar(SW) = dx*dy*pi*(-dx + 2.d0*Rmaj(i))
	A_Qbar(SE) = dx*dy*Pi*(dx + 2.d0*Rmaj(i))
	A_Qbar(NE) = dx*dy*Pi*(dx + 2.d0*Rmaj(i))
	A_Qbar(NW) = dx*dy*Pi*(-dx + 2.d0*Rmaj(i))
			!-------------------------

	rho(nx,ny) = ( rho(i,j) * vol -  &
						rho(i+1,j-1) * A_Qbar(SE) -  &
						rho(i-1,j-1) * A_Qbar(SW) -  &
						rho(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(NE)

	u(nx,ny) = ( u(i,j) * vol -  &
					  u(i+1,j-1) * A_Qbar(SE) -  &
					  u(i-1,j-1) * A_Qbar(SW) -  &
					  u(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(NE)

	v(nx,ny) = ( v(i,j) * vol -  &
					  v(i+1,j-1) * A_Qbar(SE) -  &
					  v(i-1,j-1) * A_Qbar(SW) -  &
					  v(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(NE)

	w(nx,ny) = ( w(i,j) * vol -  &
						w(i+1,j-1) * A_Qbar(SE) -  &
						w(i-1,j-1) * A_Qbar(SW) -  &
						w(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(NE)

	p(nx,ny) = ( p(i,j) * vol -  &
					  p(i+1,j-1) * A_Qbar(SE) -  &
					  p(i-1,j-1) * A_Qbar(SW) -  &
					  p(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(NE)
	p(nx,ny) = max(p(nx,ny),pmin)

!!$	bx(nx,ny) = ( bx(i,j) * vol -  &
!!$						bx(i+1,j-1) * A_Qbar(SE) -  &
!!$						bx(i-1,j-1) * A_Qbar(SW) -  &
!!$						bx(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(NE)
!!$
!!$	by(nx,ny) = ( by(i,j) * vol -  &
!!$						by(i+1,j-1) * A_Qbar(SE) -  &
!!$						by(i-1,j-1) * A_Qbar(SW) -  &
!!$						by(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(NE)
!!$
!!$	bz(nx,ny) = ( bz(i,j) * vol -  &
!!$					   bz(i+1,j-1) * A_Qbar(SE) -  &
!!$					   bz(i-1,j-1) * A_Qbar(SW) -  &
!!$					   bz(i-1,j+1) * A_Qbar(NW) ) / A_Qbar(NE)


	! ----------------------(1,ny)----------------------
	i = 2; j = ny-1

	vol = 8.d0*pi*dx*dy*Rmaj(i)

	A_Qbar(SW) = dx*dy*pi*(-dx + 2.d0*Rmaj(i))
	A_Qbar(SE) = dx*dy*Pi*(dx + 2.d0*Rmaj(i))
	A_Qbar(NE) = dx*dy*Pi*(dx + 2.d0*Rmaj(i))
	A_Qbar(NW) = dx*dy*Pi*(-dx + 2.d0*Rmaj(i))
			!-------------------------

	rho(1,ny) = ( rho(i,j) * vol -  &
						rho(i+1,j-1) * A_Qbar(SE) -  &
						rho(i-1,j-1) * A_Qbar(SW) -  &
						rho(i+1,j+1) * A_Qbar(NE) ) / A_Qbar(NW)

	u(1,ny) = ( u(i,j) * vol -  &
					  u(i+1,j-1) * A_Qbar(SE) -  &
					  u(i-1,j-1) * A_Qbar(SW) -  &
					  u(i+1,j+1) * A_Qbar(NE) ) / A_Qbar(NW)

	v(1,ny) = ( v(i,j) * vol -  &
					  v(i+1,j-1) * A_Qbar(SE) -  &
					  v(i-1,j-1) * A_Qbar(SW) -  &
					  v(i+1,j+1) * A_Qbar(NE) ) / A_Qbar(NW)

	w(1,ny) = ( w(i,j) * vol -  &
						w(i+1,j-1) * A_Qbar(SE) -  &
						w(i-1,j-1) * A_Qbar(SW) -  &
						w(i+1,j+1) * A_Qbar(NE) ) / A_Qbar(NW)

	p(1,ny) = ( p(i,j) * vol -  &
					  p(i+1,j-1) * A_Qbar(SE) -  &
					  p(i-1,j-1) * A_Qbar(SW) -  &
					  p(i+1,j+1) * A_Qbar(NE) ) / A_Qbar(NW)
	p(1,ny) = max(p(1,ny),pmin)

!!$	bx(1,ny) = ( bx(i,j) * vol -  &
!!$						bx(i+1,j-1) * A_Qbar(SE) -  &
!!$						bx(i-1,j-1) * A_Qbar(SW) -  &
!!$						bx(i+1,j+1) * A_Qbar(NE) ) / A_Qbar(NW)
!!$
!!$	by(1,ny) = ( by(i,j) * vol -  &
!!$						by(i+1,j-1) * A_Qbar(SE) -  &
!!$						by(i-1,j-1) * A_Qbar(SW) -  &
!!$						by(i+1,j+1) * A_Qbar(NE) ) / A_Qbar(NW)
!!$
!!$	bz(1,ny) = ( bz(i,j) * vol -  &
!!$					bz(i+1,j-1) * A_Qbar(SE) -  &
!!$					bz(i-1,j-1) * A_Qbar(SW) -  &
!!$					bz(i+1,j+1) * A_Qbar(NE) ) / A_Qbar(NW)



	! update the points next to the corners

	do j=1,ny, ny-1

		!!$   boundary conditions on left boundary (reflective)
		rho(0,j)=rho(1,j)
		u(0,j)=-u(1,j)
		v(0,j)=v(1,j)    !*rmaj(1)/rmaj(0)
		w(0,j)=w(1,j)     *(rmaj(1)/rmaj(0))**2
		bx(0,j)=-bx(1,j)
		by(0,j)=by(1,j)    !*rmaj(1)/rmaj(0)
		bz(0,j)=bz(1,j)*rmaj(1)/rmaj(0)

		p(0,j)=p(1,j)+(by(1,j)**2-by(0,j)**2)/  &
			(2.*rmu0)-torus*((rho(1,j)*(w(1,j)**2+w(0,j)**2)  &
			)*dx/(rmaj(1)+rmaj(0)))


		!!$   boundary conditions on right boundary (reflective)
		rho(nx+1,j)=rho(nx,j)
		u(nx+1,j)=-u(nx,j)
		v(nx+1,j)=v(nx,j)       !*rmaj(nx)/rmaj(nx+1)
		w(nx+1,j)=w(nx,j)       *(rmaj(nx)/rmaj(nx+1))**2
		bx(nx+1,j)=-bx(nx,j)
		by(nx+1,j)=by(nx,j)     !*rmaj(nx)/rmaj(nx+1)
		bz(nx+1,j)=bz(nx,j)*rmaj(nx)/rmaj(nx+1)



		p(nx+1,j)=p(nx,j)+(by(nx,j)**2-  &
		    by(nx+1,j)**2)/(2.*rmu0)+torus*(  &
		   (rho(nx,j)*(w(nx,j)**2+w(nx+1,j)**2)  &
		    )*dx/(rmaj(nx)+rmaj(nx+1)))

	enddo

	do i=1,nx, nx-1

		!!$     boundary conditions on the bottom boundary (reflective)
		rho(i,0)=rho(i,1)
		u(i,0)=u(i,1)
		v(i,0)=-v(i,1)
		w(i,0)=w(i,1)
		bx(i,0)=bx(i,1)
		by(i,0)=-by(i,1)
		bz(i,0)=bz(i,1)
		p(i,0)=p(i,1)

		!!$     boundary conditions on the top boundary (reflective)
		rho(i,ny+1)=rho(i,ny)
		u(i,ny+1)=u(i,ny)
		v(i,ny+1)=-v(i,ny)
		w(i,ny+1)=w(i,ny)
		bx(i,ny+1)=bx(i,ny)
		by(i,ny+1)=-by(i,ny)
		bz(i,ny+1)=bz(i,ny)
		p(i,ny+1)=p(i,ny)

	enddo


	return

end subroutine boundary_reflect_square_two


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
  subroutine  boundary_reflect_square_vortex_model(nx,ny,nmx,nmy,u,v,w,rho,p,bx,by,bz)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! based on ghost cell method of Dadone Grossman, comput. Fluids36, 1513 (2007)

	use mod_parameters, ONLY: dx, dy, rsh, rmu0, torus, pi, pmin, rhomin
	use mod_arrays, only : Rmaj

	integer :: nmx, nmy, nx, ny
	integer :: i, j

	real(kind=dkind) ::  rho(0:nmx+1,0:nmy+1),u(0:nmx+1,0:nmy+1),  &
      v(0:nmx+1,0:nmy+1),w(0:nmx+1,0:nmy+1),  &
      p(0:nmx+1,0:nmy+1),  &
      bx(0:nmx+1,0:nmy+1),by(0:nmx+1,0:nmy+1),bz(0:nmx+1,0:nmy+1)

	real(kind=dkind) :: vtilde2, vtilde2_ghost
	real(kind=dkind) :: ax, ay, az
	real(kind=dkind) :: small = 1.d-10

	real(kind=dkind), dimension(1:4) :: A_Qbar
	real(kind=dkind) :: vol


	do j=1,ny

		!!$   boundary conditions on left boundary (reflective)

		! use standard approach for magnetic field

		bx(0,j) = -bx(1,j)
		by(0,j) = by(1,j)    !*rmaj(1)/rmaj(0)
		bz(0,j) = bz(1,j)*rmaj(1)/rmaj(0)

		! get the rest using the vortex model and entropy, enthalpy condition

		u(0,j)=-u(1,j)

		vtilde2 = v(1,j)**2 + w(1,j)**2

		p(0,j)=p(1,j)+(by(1,j)**2-by(0,j)**2)/  &
			(2.d0*rmu0) - 2.d0*torus*((rho(1,j)*vtilde2  &
			)*dx/(rmaj(1)+rmaj(0)))
		p(0,j) = max(pmin,p(0,j))

		rho(0,j) = rho(1,j) * (p(0,j)/p(1,j))**(1.d0/rsh)

		vtilde2_ghost = vtilde2 + 2.d0*rsh/(rsh-1.d0) * (p(1,j)/rho(1,j) - p(0,j)/rho(0,j))

		if((abs(v(1,j))+abs(w(1,j))<small).or.(vtilde2_ghost<=0.d0)) then

			v(0,j) = 0.d0; w(0,j) = 0.d0

		else

			ay = v(1,j)/max(sqrt(v(1,j)**2+w(1,j)**2),small)
			az = w(1,j)/max(sqrt(v(1,j)**2+w(1,j)**2),small)

			v(0,j) = ay/sqrt(ay**4+az**4) * sqrt(vtilde2_ghost)
			w(0,j) = az/sqrt(ay**4+az**4) * sqrt(vtilde2_ghost)

		endif

		!!$   boundary conditions on right boundary (reflective)

		! use standard approach for magnetic field

		bx(nx+1,j)=-bx(nx,j)
		by(nx+1,j)=by(nx,j)     !*rmaj(nx)/rmaj(nx+1)
		bz(nx+1,j)=bz(nx,j)*rmaj(nx)/rmaj(nx+1)

		! get the rest using the vortex model and entropy, enthalpy condition

		u(nx+1,j)=-u(nx,j)

		vtilde2 = v(nx,j)**2 + w(nx,j)**2

		p(nx+1,j)=p(nx,j)+(by(nx,j)**2-by(nx+1,j)**2)/  &
			(2.d0*rmu0) + 2.d0*torus*((rho(nx,j)*vtilde2  &
			)*dx/(rmaj(nx)+rmaj(nx+1)))
		p(nx+1,j) = max(pmin,p(nx+1,j))

		rho(nx+1,j) = rho(nx,j) * (p(nx+1,j)/p(nx,j))**(1.d0/rsh)

		vtilde2_ghost = vtilde2 + 2.d0*rsh/(rsh-1.d0) * (p(nx,j)/rho(nx,j) - p(nx+1,j)/rho(nx+1,j))

		if((abs(v(nx,j))+abs(w(nx,j))<small).or.(vtilde2_ghost<=0.d0)) then

			v(nx+1,j) = 0.d0; w(nx+1,j) = 0.d0

		else

			ay = v(nx,j)/max(sqrt(v(nx,j)**2+w(nx,j)**2),small)
			az = w(nx,j)/max(sqrt(v(nx,j)**2+w(nx,j)**2),small)

			v(nx+1,j) = ay/sqrt(ay**4+az**4) * sqrt(vtilde2_ghost)
			w(nx+1,j) = az/sqrt(ay**4+az**4) * sqrt(vtilde2_ghost)

		endif

	enddo




	do i=1,nx

		!!$     boundary conditions on the bottom boundary (reflective)

		! use standard approach for magnetic field

		bx(i,0) = bx(i,1)
		by(i,0) = -by(i,1)
		bz(i,0) = bz(i,1)

		! get the rest using the vortex model and entropy, enthalpy condition

		v(i,0) = -v(i,1)

		vtilde2 = u(i,1)**2 + w(i,1)**2

		p(i,0) = p(i,1)
		p(i,0) = max(pmin,p(i,0))

		rho(i,0) = rho(i,1)* (p(i,0)/p(i,1))**(1.d0/rsh)

		vtilde2_ghost = vtilde2 + 2.d0*rsh/(rsh-1.d0) * (p(i,1)/rho(i,1) - p(i,0)/rho(i,0))

		if((abs(u(i,1))+abs(w(i,1))<small).or.(vtilde2_ghost<=0.d0)) then

			u(i,0) = 0.d0; w(i,0) = 0.d0

		else

			ax = u(i,1)/max(sqrt(u(i,1)**2+w(i,1)**2),small)
			az = w(i,1)/max(sqrt(u(i,1)**2+w(i,1)**2),small)

			u(i,0) = ax/sqrt(ax**4+az**4) * sqrt(vtilde2_ghost)
			w(i,0) = az/sqrt(ax**4+az**4) * sqrt(vtilde2_ghost)

		endif

		

		!!$     boundary conditions on the top boundary (reflective)

		! use standard approach for magnetic field

		bx(i,ny+1) = bx(i,ny)
		by(i,ny+1) = -by(i,ny)
		bz(i,ny+1) = bz(i,ny)

		! get the rest using the vortex model and entropy, enthalpy condition

		v(i,ny+1) = -v(i,ny)

		vtilde2 = u(i,ny)**2 + w(i,ny)**2

		p(i,ny+1)=p(i,ny)
		p(i,ny+1) = max(pmin,p(i,ny+1))

		rho(i,ny+1) = rho(i,ny)* (p(i,ny+1)/p(i,ny))**(1.d0/rsh)

		vtilde2_ghost = vtilde2 + 2.d0*rsh/(rsh-1.d0) * (p(i,ny)/rho(i,ny) - p(i,ny+1)/rho(i,ny+1))

		if((abs(u(i,1))+abs(w(i,1))<small).or.(vtilde2_ghost<=0.d0)) then

			u(i,ny+1) = 0.d0; w(i,ny+1) = 0.d0

		else

!			if(i==nx) then
!				print*, i
!				print*, u(i,ny),w(i,ny), vtilde2_ghost
!				print*, rho(i,ny+1), p(i,ny+1)
!			endif

			ax = u(i,ny)/max(sqrt(u(i,ny)**2+w(i,ny)**2),small)
			az = w(i,ny)/max(sqrt(u(i,ny)**2+w(i,ny)**2),small)

!			if(i==nx) then
!				print*, ax,az
!				print*, '    ----     '
!			endif

			u(i,ny+1) = ax/sqrt(ax**4+az**4) * sqrt(vtilde2_ghost)
			w(i,ny+1) = az/sqrt(ax**4+az**4) * sqrt(vtilde2_ghost)

		endif


	enddo

	return

end subroutine boundary_reflect_square_vortex_model

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine  boundary_vacuum_field_copy(bxpr, bypr, bzpr)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! this routine copies the field to the predictor variables:
! in the tied-field version, the field on the boundary is fixed!

	use mod_parameters, only : nx, ny
	use mod_arrays, only : bx, by, bz

	real(kind=dkind), dimension(0:nx+1,0:ny+1) :: bxpr, bypr, bzpr

	integer :: i, j

	do j = 0, ny+1

		i = 0

		bxpr(i,j) = bx(i,j)
		bypr(i,j) = by(i,j)
		bzpr(i,j) = bz(i,j)

		i = nx+1

		bxpr(i,j) = bx(i,j)
		bypr(i,j) = by(i,j)
		bzpr(i,j) = bz(i,j)

	enddo

	do i = 0, nx+1

		j = 0

		bxpr(i,j) = bx(i,j)
		bypr(i,j) = by(i,j)
		bzpr(i,j) = bz(i,j)

		j = ny+1

		bxpr(i,j) = bx(i,j)
		bypr(i,j) = by(i,j)
		bzpr(i,j) = bz(i,j)

	enddo

	continue

	return

end subroutine  boundary_vacuum_field_copy

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine  boundary_vacuum_0(nx,ny,nmx,nmy,u,v,w,rho,p,bx,by,bz)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	use mod_parameters, ONLY: dx, rsh, rmu0, torus, pmin, rhomin
	use mod_arrays, only : Rmaj

	integer :: nmx, nmy, nx, ny
	integer :: i, j

	real(kind=dkind) ::  rho(0:nmx+1,0:nmy+1),u(0:nmx+1,0:nmy+1),  &
      v(0:nmx+1,0:nmy+1),w(0:nmx+1,0:nmy+1),  &
      p(0:nmx+1,0:nmy+1),  &
      bx(0:nmx+1,0:nmy+1),by(0:nmx+1,0:nmy+1),bz(0:nmx+1,0:nmy+1)

	real(kind=dkind), save :: CS0 = 0.d0
	real(kind=dkind) :: csloc, vploc

	if(CS0==0.d0) CS0 = sqrt(rsh*pmin/rhomin)

	do j=1,ny
		!!$   boundary conditions on left boundary (inflow/outlfow)

		i = 1

		csloc = sqrt(rsh*p(i,j)/rho(i,j))*sqrt((bx(i,j)**2+by(i,j)**2)/(bx(i,j)**2+by(i,j)**2+bz(i,j)**2))
		vploc = sqrt(u(i,j)**2+v(i,j)**2)

		!distinguish by the character of the point (inflow/outflow)

		if(u(i,j)>=0.d0) then
		! inflow case
			if(vploc<csloc) then
				call bc_v_0_inlet_sub(i,j,-1)
			else
				call bc_v_0_inlet_super(i,j,-1)
			endif
		else
		! outflow case
			if(vploc<csloc) then
				call bc_v_0_outlet_sub(i,j,-1)
			else
				call bc_v_0_outlet_super(i,j,-1)
			endif
		endif

		!!$   boundary conditions on right boundary (inflow/outlfow)

		i = nx

		csloc = sqrt(rsh*p(i,j)/rho(i,j))*sqrt((bx(i,j)**2+by(i,j)**2)/(bx(i,j)**2+by(i,j)**2+bz(i,j)**2))
		vploc = sqrt(u(i,j)**2+v(i,j)**2)

		!distinguish by the character of the point (inflow/outflow)

		if(u(i,j)<=0.d0) then
		! inflow case
			if(vploc<csloc) then
				call bc_v_0_inlet_sub(i,j,1)
			else
				call bc_v_0_inlet_super(i,j,1)
			endif
		else
		! outflow case
			if(vploc<csloc) then
				call bc_v_0_outlet_sub(i,j,1)
			else
				call bc_v_0_outlet_super(i,j,1)
			endif
		endif

	enddo




	do i=1,nx

		!!$     boundary conditions on the bottom boundary (inflow/outlfow)

		j = 1

		csloc = sqrt(rsh*p(i,j)/rho(i,j))*sqrt((bx(i,j)**2+by(i,j)**2)/(bx(i,j)**2+by(i,j)**2+bz(i,j)**2))
		vploc = sqrt(u(i,j)**2+v(i,j)**2)

		!distinguish by the character of the point (inflow/outflow)

		if(v(i,j)>=0.d0) then
		! inflow case
			if(vploc<csloc) then
				call bc_v_0_inlet_sub(i,j,-2)
			else
				call bc_v_0_inlet_super(i,j,-2)
			endif
		else
		! outflow case
			if(vploc<csloc) then
				call bc_v_0_outlet_sub(i,j,-2)
			else
				call bc_v_0_outlet_super(i,j,-2)
			endif
		endif

		!!$     boundary conditions on the top boundary (inflow/outlfow)

		j = ny

		csloc = sqrt(rsh*p(i,j)/rho(i,j))*sqrt((bx(i,j)**2+by(i,j)**2)/(bx(i,j)**2+by(i,j)**2+bz(i,j)**2))
		vploc = sqrt(u(i,j)**2+v(i,j)**2)

		!distinguish by the character of the point (inflow/outflow)

		if(v(i,j)<=0.d0) then
		! inflow case
			if(vploc<csloc) then
				call bc_v_0_inlet_sub(i,j,2)
			else
				call bc_v_0_inlet_super(i,j,2)
			endif
		else
		! outflow case
			if(vploc<csloc) then
				call bc_v_0_outlet_sub(i,j,2)
			else
				call bc_v_0_outlet_super(i,j,2)
			endif
		endif

	enddo

	return

	contains

	!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	subroutine bc_v_0_inlet_sub(i,j,dir)
	!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	! minus sign in dir stands for left/bottom, plus for right/top

		integer :: dir
		integer :: i, j
		integer :: i_BC, j_BC
		real(kind=dkind) :: b_edge

		if(abs(dir)==1) then
		! LEFT/RIGHT boundary

			i_BC = i+dir; j_BC = j

			b_edge = sqrt(bx(i_BC,j_BC)**2+by(i_BC,j_BC)**2+bz(i_BC,j_BC)**2)

			! assign constant rho
			rho(i_BC,j_BC) = rho_boundary

			! assign u, v (arbitrarily)
			u(i_BC,j_BC) = Mach_boundary * CS0 * abs(bx(i_BC,j_BC)/b_edge) * (-dir)
			v(i_BC,j_BC) = Mach_boundary * CS0 * abs(by(i_BC,j_BC)/b_edge) * sign(1.d0,v(i,j))

			! extrapolate w
			w(i_BC,j_BC) = w(i,j)*(Rmaj(i)/Rmaj(i_BC))**2

			! extrapolate pressure
			p(i_BC,j_BC) = p(i,j) + (by(i,j)**2-by(i_BC,j_BC)**2)/(2.*rmu0) +   &
				dir*torus*( (rho(i,j)*(w(i,j)**2+w(i_BC,j_BC)**2) )*dx/(Rmaj(i)+rmaj(i_BC)))

		elseif(abs(dir)==2) then
		! TOP/BOTTOM boundary

			i_BC = i; j_BC = j+dir/2

			b_edge = sqrt(bx(i_BC,j_BC)**2+by(i_BC,j_BC)**2+bz(i_BC,j_BC)**2)

			! assign constant rho
			rho(i_BC,j_BC) = rho_boundary

			! assign u, v (arbitrarily)
			u(i_BC,j_BC) = Mach_boundary * CS0 * abs(bx(i_BC,j_BC)/b_edge) * sign(1.d0,u(i,j))
			v(i_BC,j_BC) = Mach_boundary * CS0 * abs(by(i_BC,j_BC)/b_edge) * (-dir/2.d0)

			! extrapolate w
			w(i_BC,j_BC) = w(i,j)

			! extrapolate pressure
			p(i_BC,j_BC) = p(i,j)

		endif

		p(i_BC,j_BC) = max(pmin,p(i_BC,j_BC))

	end subroutine bc_v_0_inlet_sub

	!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	subroutine bc_v_0_inlet_super(i,j,dir)
	!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	! minus sign in dir stands for left/bottom, plus for right/top

		integer :: dir
		integer :: i, j
		integer :: i_BC, j_BC
		real(kind=dkind) :: b_edge

		if(abs(dir)==1) then
		! LEFT/RIGHT boundary

			i_BC = i+dir; j_BC = j

			b_edge = sqrt(bx(i_BC,j_BC)**2+by(i_BC,j_BC)**2+bz(i_BC,j_BC)**2)

			! assign constant rho
			rho(i_BC,j_BC) = rho_boundary

			! assign u, v (arbitrarily)
			u(i_BC,j_BC) = (1.d0+Mach_boundary) * CS0 * abs(bx(i_BC,j_BC)/b_edge) * (-dir)
			v(i_BC,j_BC) = (1.d0+Mach_boundary) * CS0 * abs(by(i_BC,j_BC)/b_edge) * sign(1.d0,v(i,j))

			! assign w (arbitrarily)
			w(i_BC,j_BC) = 0.d0

			! assign pressure
			p(i_BC,j_BC) = p_boundary

		elseif(abs(dir)==2) then
		! TOP/BOTTOM boundary

			i_BC = i; j_BC = j+dir/2

			b_edge = sqrt(bx(i_BC,j_BC)**2+by(i_BC,j_BC)**2+bz(i_BC,j_BC)**2)

			! assign constant rho
			rho(i_BC,j_BC) = rho_boundary

			! assign u, v (arbitrarily)
			u(i_BC,j_BC) = (1.d0+Mach_boundary) * CS0 * abs(bx(i_BC,j_BC)/b_edge) * sign(1.d0,u(i,j))
			v(i_BC,j_BC) = (1.d0+Mach_boundary) * CS0 * abs(by(i_BC,j_BC)/b_edge) * (-dir/2.d0)

			! assign w (arbitrarily)
			w(i_BC,j_BC) = 0.d0

			! assign pressure
			p(i_BC,j_BC) = p_boundary

		endif

	end subroutine bc_v_0_inlet_super

	!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	subroutine bc_v_0_outlet_sub(i,j,dir)
	!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	! minus sign in dir stands for left/bottom, plus for right/top

		integer :: dir
		integer :: i, j
		integer :: i_BC, j_BC

		if(abs(dir)==1) then
		! LEFT/RIGHT boundary

			i_BC = i+dir; j_BC = j

			! extrapolate rho
			rho(i_BC,j_BC) = rho(i,j)

			! extrapolate u, v, w
			u(i_BC,j_BC) = u(i,j)
			v(i_BC,j_BC) = v(i,j)
			w(i_BC,j_BC) = w(i,j)*(Rmaj(i)/Rmaj(i_BC))**2

			! extrapolate pressure
			p(i_BC,j_BC) = p(i,j) + (by(i,j)**2-by(i_BC,j_BC)**2)/(2.*rmu0) +   &
				dir*torus*( (rho(i,j)*(w(i,j)**2+w(i_BC,j_BC)**2) )*dx/(Rmaj(i)+rmaj(i_BC)))

			! pressure in principle should be assigned, use p_boundary?

		elseif(abs(dir)==2) then
		! TOP/BOTTOM boundary

			i_BC = i; j_BC = j+dir/2

			! extrapolate rho
			rho(i_BC,j_BC) = rho(i,j)

			! extrapolate u, v, w
			u(i_BC,j_BC) = u(i,j)
			v(i_BC,j_BC) = v(i,j)
			w(i_BC,j_BC) = w(i,j)

			! extrapolate pressure
			p(i_BC,j_BC) = p(i,j)

			! pressure in principle should be assigned, use p_boundary?

		endif

		rho(i_BC,j_BC) = max(rhomin,rho(i_BC,j_BC))
		p(i_BC,j_BC) = max(pmin,p(i_BC,j_BC))

	end subroutine bc_v_0_outlet_sub

	!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	subroutine bc_v_0_outlet_super(i,j,dir)
	!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	! minus sign in dir stands for left/bottom, plus for right/top

		integer :: dir
		integer :: i, j
		integer :: i_BC, j_BC

		if(abs(dir)==1) then
		! LEFT/RIGHT boundary

			i_BC = i+dir; j_BC = j

			! extrapolate rho
			rho(i_BC,j_BC) = rho(i,j)

			! extrapolate u, v, w
			u(i_BC,j_BC) = u(i,j)
			v(i_BC,j_BC) = v(i,j)
			w(i_BC,j_BC) = w(i,j)*(Rmaj(i)/Rmaj(i_BC))**2

			! extrapolate pressure
			p(i_BC,j_BC) = p(i,j) + (by(i,j)**2-by(i_BC,j_BC)**2)/(2.*rmu0) +   &
				dir*torus*( (rho(i,j)*(w(i,j)**2+w(i_BC,j_BC)**2) )*dx/(Rmaj(i)+rmaj(i_BC)))

		elseif(abs(dir)==2) then
		! TOP/BOTTOM boundary

			i_BC = i; j_BC = j+dir/2

			! extrapolate rho
			rho(i_BC,j_BC) = rho(i,j)

			! extrapolate u, v, w
			u(i_BC,j_BC) = u(i,j)
			v(i_BC,j_BC) = v(i,j)
			w(i_BC,j_BC) = w(i,j)

			! extrapolate pressure
			p(i_BC,j_BC) = p(i,j)

		endif

		rho(i_BC,j_BC) = max(rhomin,rho(i_BC,j_BC))
		p(i_BC,j_BC) = max(pmin,p(i_BC,j_BC))

	end subroutine bc_v_0_outlet_super

end subroutine boundary_vacuum_0


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine  boundary_vacuum_1(nx,ny,nmx,nmy,u,v,w,rho,p,bx,by,bz)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! similar to previous, some difference in extrapolations/assignements

	use mod_parameters, ONLY: dx, rsh, rmu0, torus, pmin, rhomin
	use mod_arrays, only : Rmaj

	integer :: nmx, nmy, nx, ny
	integer :: i, j

	real(kind=dkind) ::  rho(0:nmx+1,0:nmy+1),u(0:nmx+1,0:nmy+1),  &
      v(0:nmx+1,0:nmy+1),w(0:nmx+1,0:nmy+1),  &
      p(0:nmx+1,0:nmy+1),  &
      bx(0:nmx+1,0:nmy+1),by(0:nmx+1,0:nmy+1),bz(0:nmx+1,0:nmy+1)

	real(kind=dkind), save :: CS0 = 0.d0
	real(kind=dkind) :: csloc, vploc

	if(CS0==0.d0) CS0 = sqrt(rsh*pmin/rhomin)

	do j=1,ny
		!!$   boundary conditions on left boundary (inflow/outlfow)

		i = 1

		csloc = sqrt(rsh*p(i,j)/rho(i,j))*sqrt((bx(i,j)**2+by(i,j)**2)/(bx(i,j)**2+by(i,j)**2+bz(i,j)**2))
		vploc = sqrt(u(i,j)**2+v(i,j)**2)

		!distinguish by the character of the point (inflow/outflow)

		if(u(i,j)>=0.d0) then
		! inflow case
			if(vploc<csloc) then
				call bc_v_1_inlet_sub(i,j,-1)
			else
				call bc_v_1_inlet_super(i,j,-1)
			endif
		else
		! outflow case
			if(vploc<csloc) then
				call bc_v_1_outlet_sub(i,j,-1)
			else
				call bc_v_1_outlet_super(i,j,-1)
			endif
		endif

		!!$   boundary conditions on right boundary (inflow/outlfow)

		i = nx

		csloc = sqrt(rsh*p(i,j)/rho(i,j))*sqrt((bx(i,j)**2+by(i,j)**2)/(bx(i,j)**2+by(i,j)**2+bz(i,j)**2))
		vploc = sqrt(u(i,j)**2+v(i,j)**2)

		!distinguish by the character of the point (inflow/outflow)

		if(u(i,j)<=0.d0) then
		! inflow case
			if(vploc<csloc) then
				call bc_v_1_inlet_sub(i,j,1)
			else
				call bc_v_1_inlet_super(i,j,1)
			endif
		else
		! outflow case
			if(vploc<csloc) then
				call bc_v_1_outlet_sub(i,j,1)
			else
				call bc_v_1_outlet_super(i,j,1)
			endif
		endif

	enddo




	do i=1,nx

		!!$     boundary conditions on the bottom boundary (inflow/outlfow)

		j = 1

		csloc = sqrt(rsh*p(i,j)/rho(i,j))*sqrt((bx(i,j)**2+by(i,j)**2)/(bx(i,j)**2+by(i,j)**2+bz(i,j)**2))
		vploc = sqrt(u(i,j)**2+v(i,j)**2)

		!distinguish by the character of the point (inflow/outflow)

		if(v(i,j)>=0.d0) then
		! inflow case
			if(vploc<csloc) then
				call bc_v_1_inlet_sub(i,j,-2)
			else
				call bc_v_1_inlet_super(i,j,-2)
			endif
		else
		! outflow case
			if(vploc<csloc) then
				call bc_v_1_outlet_sub(i,j,-2)
			else
				call bc_v_1_outlet_super(i,j,-2)
			endif
		endif

		!!$     boundary conditions on the top boundary (inflow/outlfow)

		j = ny

		csloc = sqrt(rsh*p(i,j)/rho(i,j))*sqrt((bx(i,j)**2+by(i,j)**2)/(bx(i,j)**2+by(i,j)**2+bz(i,j)**2))
		vploc = sqrt(u(i,j)**2+v(i,j)**2)

		!distinguish by the character of the point (inflow/outflow)

		if(v(i,j)<=0.d0) then
		! inflow case
			if(vploc<csloc) then
				call bc_v_1_inlet_sub(i,j,2)
			else
				call bc_v_1_inlet_super(i,j,2)
			endif
		else
		! outflow case
			if(vploc<csloc) then
				call bc_v_1_outlet_sub(i,j,2)
			else
				call bc_v_1_outlet_super(i,j,2)
			endif
		endif

	enddo

	return

	contains

	!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	subroutine bc_v_1_inlet_sub(i,j,dir)
	!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	! minus sign in dir stands for left/bottom, plus for right/top

		integer :: dir
		integer :: i, j
		integer :: i_BC, j_BC
		real(kind=dkind) :: b_edge

		if(abs(dir)==1) then
		! LEFT/RIGHT boundary

			i_BC = i+dir; j_BC = j

			b_edge = sqrt(bx(i_BC,j_BC)**2+by(i_BC,j_BC)**2+bz(i_BC,j_BC)**2)

			! assign constant rho
			rho(i_BC,j_BC) = rho_boundary

			! assign u, v (arbitrarily)
			u(i_BC,j_BC) = Mach_boundary * CS0 * abs(bx(i_BC,j_BC)/b_edge) * (-dir)
			v(i_BC,j_BC) = Mach_boundary * CS0 * abs(by(i_BC,j_BC)/b_edge) * sign(1.d0,v(i,j))

			! extrapolate w
			w(i_BC,j_BC) = w(i,j)*(Rmaj(i)/Rmaj(i_BC))**2

			! extrapolate pressure
			p(i_BC,j_BC) = p_boundary

		elseif(abs(dir)==2) then
		! TOP/BOTTOM boundary

			i_BC = i; j_BC = j+dir/2

			b_edge = sqrt(bx(i_BC,j_BC)**2+by(i_BC,j_BC)**2+bz(i_BC,j_BC)**2)

			! assign constant rho
			rho(i_BC,j_BC) = rho_boundary

			! assign u, v (arbitrarily)
			u(i_BC,j_BC) = Mach_boundary * CS0 * abs(bx(i_BC,j_BC)/b_edge) * sign(1.d0,u(i,j))
			v(i_BC,j_BC) = Mach_boundary * CS0 * abs(by(i_BC,j_BC)/b_edge) * (-dir/2.d0)

			! extrapolate w
			w(i_BC,j_BC) = w(i,j)

			! extrapolate pressure
			p(i_BC,j_BC) = p(i,j)

		endif

		p(i_BC,j_BC) = max(pmin,p(i_BC,j_BC))

	end subroutine bc_v_1_inlet_sub

	!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	subroutine bc_v_1_inlet_super(i,j,dir)
	!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	! minus sign in dir stands for left/bottom, plus for right/top

		integer :: dir
		integer :: i, j
		integer :: i_BC, j_BC
		real(kind=dkind) :: b_edge

		if(abs(dir)==1) then
		! LEFT/RIGHT boundary

			i_BC = i+dir; j_BC = j

			b_edge = sqrt(bx(i_BC,j_BC)**2+by(i_BC,j_BC)**2+bz(i_BC,j_BC)**2)

			! assign constant rho
			rho(i_BC,j_BC) = rho_boundary

			! assign u, v (arbitrarily)
			u(i_BC,j_BC) = (1.d0+Mach_boundary) * CS0 * abs(bx(i_BC,j_BC)/b_edge) * (-dir)
			v(i_BC,j_BC) = (1.d0+Mach_boundary) * CS0 * abs(by(i_BC,j_BC)/b_edge) * sign(1.d0,v(i,j))

			! assign w (arbitrarily)
			w(i_BC,j_BC) = 0.d0

			! assign pressure
			p(i_BC,j_BC) = p_boundary

		elseif(abs(dir)==2) then
		! TOP/BOTTOM boundary

			i_BC = i; j_BC = j+dir/2

			b_edge = sqrt(bx(i_BC,j_BC)**2+by(i_BC,j_BC)**2+bz(i_BC,j_BC)**2)

			! assign constant rho
			rho(i_BC,j_BC) = rho_boundary

			! assign u, v (arbitrarily)
			u(i_BC,j_BC) = (1.d0+Mach_boundary) * CS0 * abs(bx(i_BC,j_BC)/b_edge) * sign(1.d0,u(i,j))
			v(i_BC,j_BC) = (1.d0+Mach_boundary) * CS0 * abs(by(i_BC,j_BC)/b_edge) * (-dir/2.d0)

			! assign w (arbitrarily)
			w(i_BC,j_BC) = 0.d0

			! assign pressure
			p(i_BC,j_BC) = p_boundary

		endif

	end subroutine bc_v_1_inlet_super

	!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	subroutine bc_v_1_outlet_sub(i,j,dir)
	!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	! minus sign in dir stands for left/bottom, plus for right/top

		integer :: dir
		integer :: i, j
		integer :: i_BC, j_BC

		if(abs(dir)==1) then
		! LEFT/RIGHT boundary

			i_BC = i+dir; j_BC = j

			! extrapolate rho
			rho(i_BC,j_BC) = rho(i,j)

			! extrapolate u, v, w
			u(i_BC,j_BC) = u(i,j)
			v(i_BC,j_BC) = v(i,j)
			w(i_BC,j_BC) = w(i,j)*(Rmaj(i)/Rmaj(i_BC))**2

			! extrapolate pressure
			p(i_BC,j_BC) = p_boundary

			! pressure in principle should be assigned, use p_boundary?

		elseif(abs(dir)==2) then
		! TOP/BOTTOM boundary

			i_BC = i; j_BC = j+dir/2

			! extrapolate rho
			rho(i_BC,j_BC) = rho(i,j)

			! extrapolate u, v, w
			u(i_BC,j_BC) = u(i,j)
			v(i_BC,j_BC) = v(i,j)
			w(i_BC,j_BC) = w(i,j)

			! extrapolate pressure
			p(i_BC,j_BC) = p(i,j)

			! pressure in principle should be assigned, use p_boundary?

		endif

		rho(i_BC,j_BC) = max(rhomin,rho(i_BC,j_BC))
		p(i_BC,j_BC) = max(pmin,p(i_BC,j_BC))

	end subroutine bc_v_1_outlet_sub

	!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	subroutine bc_v_1_outlet_super(i,j,dir)
	!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	! minus sign in dir stands for left/bottom, plus for right/top

		integer :: dir
		integer :: i, j
		integer :: i_BC, j_BC

		if(abs(dir)==1) then
		! LEFT/RIGHT boundary

			i_BC = i+dir; j_BC = j

			! extrapolate rho
			rho(i_BC,j_BC) = rho(i,j)

			! extrapolate u, v, w
			u(i_BC,j_BC) = u(i,j)
			v(i_BC,j_BC) = v(i,j)
			w(i_BC,j_BC) = w(i,j)*(Rmaj(i)/Rmaj(i_BC))**2

			! extrapolate pressure
			p(i_BC,j_BC) = p(i,j)

		elseif(abs(dir)==2) then
		! TOP/BOTTOM boundary

			i_BC = i; j_BC = j+dir/2

			! extrapolate rho
			rho(i_BC,j_BC) = rho(i,j)

			! extrapolate u, v, w
			u(i_BC,j_BC) = u(i,j)
			v(i_BC,j_BC) = v(i,j)
			w(i_BC,j_BC) = w(i,j)

			! extrapolate pressure
			p(i_BC,j_BC) = p(i,j)

		endif

		rho(i_BC,j_BC) = max(rhomin,rho(i_BC,j_BC))
		p(i_BC,j_BC) = max(pmin,p(i_BC,j_BC))

	end subroutine bc_v_1_outlet_super

end subroutine boundary_vacuum_1

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine  boundary_vacuum_vn0(nx,ny,nmx,nmy,u,v,w,rho,p,bx,by,bz)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! v_normal is set to zero (same as in no-vacuum case), B is left assigned

	use mod_parameters, ONLY: dx, rsh, rmu0, torus, pmin, rhomin
	use mod_arrays, only : Rmaj

	integer :: nmx, nmy, nx, ny
	integer :: i, j

	real(kind=dkind) ::  rho(0:nmx+1,0:nmy+1),u(0:nmx+1,0:nmy+1),  &
      v(0:nmx+1,0:nmy+1),w(0:nmx+1,0:nmy+1),  &
      p(0:nmx+1,0:nmy+1),  &
      bx(0:nmx+1,0:nmy+1),by(0:nmx+1,0:nmy+1),bz(0:nmx+1,0:nmy+1)



	do j=1,ny

		!!$   boundary conditions on left boundary (reflective)
		rho(0,j) = min(rho_boundary,rho(1,j))
		u(0,j)=-u(1,j)
		v(0,j)=v(1,j)    !*rmaj(1)/rmaj(0)
		w(0,j)=w(1,j)     *(rmaj(1)/rmaj(0))**2
!!$		bx(0,j)=-bx(1,j)
!!$		by(0,j)=by(1,j)    !*rmaj(1)/rmaj(0)
!!$		bz(0,j)=bz(1,j)*rmaj(1)/rmaj(0)

		p(0,j)=p(1,j)+(by(1,j)**2-by(0,j)**2)/  &
			(2.*rmu0)-torus*((rho(1,j)*(w(1,j)**2+w(0,j)**2)  &
			)*dx/(rmaj(1)+rmaj(0)))
		p(0,j) = max(pmin,p(0,j))

		!!$   boundary conditions on right boundary (reflective)
		rho(nx+1,j) = min(rho(nx,j),rho_boundary)
		u(nx+1,j)=-u(nx,j)
		v(nx+1,j)=v(nx,j)       !*rmaj(nx)/rmaj(nx+1)
		w(nx+1,j)=w(nx,j)       *(rmaj(nx)/rmaj(nx+1))**2
!!$		bx(nx+1,j)=-bx(nx,j)
!!$		by(nx+1,j)=by(nx,j)     !*rmaj(nx)/rmaj(nx+1)
!!$		bz(nx+1,j)=bz(nx,j)*rmaj(nx)/rmaj(nx+1)

		p(nx+1,j)=p(nx,j)+(by(nx,j)**2-  &
		    by(nx+1,j)**2)/(2.*rmu0)+torus*(  &
		   (rho(nx,j)*(w(nx,j)**2+w(nx+1,j)**2)  &
		    )*dx/(rmaj(nx)+rmaj(nx+1)))
		p(nx+1,j) = max(pmin,p(nx+1,j))

	enddo




	do i=1,nx

		!!$     boundary conditions on the bottom boundary (reflective)
		rho(i,0) = min(rho(i,1),rho_boundary)
		u(i,0)=u(i,1)
		v(i,0)=-v(i,1)
		w(i,0)=w(i,1)
!!$		bx(i,0)=bx(i,1)
!!$		by(i,0)=-by(i,1)
!!$		bz(i,0)=bz(i,1)
		p(i,0)=p(i,1)

		!!$     boundary conditions on the top boundary (reflective)
		rho(i,ny+1) = min(rho_boundary,rho(i,ny))
		u(i,ny+1)=u(i,ny)
		v(i,ny+1)=-v(i,ny)
		w(i,ny+1)=w(i,ny)
!!$		bx(i,ny+1)=bx(i,ny)
!!$		by(i,ny+1)=-by(i,ny)
!!$		bz(i,ny+1)=bz(i,ny)
		p(i,ny+1)=p(i,ny)

	enddo

	return

end subroutine boundary_vacuum_vn0


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine  boundary_vacuum_CS(u,v,w,rho,p,bx,by,bz)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! sonic outflow at the edge

	use mod_parameters, ONLY: nx, ny, dx, rsh, rmu0, torus, pmin, rhomin
	use mod_arrays, only : nmx, nmy, Rmaj

	real(kind=dkind), dimension(0:nmx+1,0:nmy+1) :: rho, u, v, w, p, bx, by, bz
	real(kind=dkind) :: modB, Cs, dir
	integer :: i, j

	do j=1,ny

		!!$   boundary conditions on left boundary (sonic outflow)

		modB = sqrt(bx(0,j)**2 + by(0,j)**2 + bz(0,j)**2)
		Cs = sqrt(rsh*p(1,j)/rho(1,j))
		dir = -sign(1.d0,bx(0,j))

		rho(0,j) = max(rho(1,j),rhomin)

		u(0,j) = Mach_boundary * Cs * dir*bx(0,j)/modB
		v(0,j) = Mach_boundary * Cs * dir*by(0,j)/modB
		w(0,j) = Mach_boundary * Cs * dir*bz(0,j)/modB

		p(0,j)=p(1,j)+(by(1,j)**2-by(0,j)**2)/  &
			(2.d0*rmu0)-torus*((rho(1,j)*(w(1,j)**2+w(0,j)**2)  &
			)*dx/(rmaj(1)+rmaj(0)))
		p(0,j) = max(pmin,p(0,j))

		!!$   boundary conditions on right boundary (sonic outflow)

		modB = sqrt(bx(nx+1,j)**2 + by(nx+1,j)**2 + bz(nx+1,j)**2)
		Cs = sqrt(rsh*p(nx,j)/rho(nx,j))
		dir = sign(1.d0,bx(nx+1,j))

		rho(nx+1,j) = max(rho(nx,j),rhomin)

		u(nx+1,j) = Mach_boundary * Cs * dir*bx(nx+1,j)/modB
		v(nx+1,j) = Mach_boundary * Cs * dir*by(nx+1,j)/modB
		w(nx+1,j) = Mach_boundary * Cs * dir*bz(nx+1,j)/modB

		p(nx+1,j)=p(nx,j)+(by(nx,j)**2-  &
		    by(nx+1,j)**2)/(2.*rmu0)+torus*(  &
		   (rho(nx,j)*(w(nx,j)**2+w(nx+1,j)**2)  &
		    )*dx/(rmaj(nx)+rmaj(nx+1)))
		p(nx+1,j) = max(pmin,p(nx+1,j))

	enddo




	do i=1,nx

		!!$     boundary conditions on the bottom boundary (sonic outflow)

		modB = sqrt(bx(i,0)**2 + by(i,0)**2 + bz(i,0)**2)
		Cs = sqrt(rsh*p(i,1)/rho(i,1))
		dir = -sign(1.d0,by(i,0))

		rho(i,0) = max(rho(i,1),rhomin)

		u(i,0) = Mach_boundary * Cs * dir*bx(i,0)/modB
		v(i,0) = Mach_boundary * Cs * dir*by(i,0)/modB
		w(i,0) = Mach_boundary * Cs * dir*bz(i,0)/modB

!		u(i,0) = Mach_boundary * Cs * bx(i,0)/modB * (-by(i,0)/max(abs(by(i,0).1.d-8))
!		v(i,0) = Mach_boundary * Cs * by(i,0)/modB * by(i,0)/max(abs(by(i,0).1.d-8)
!		w(i,0) = Mach_boundary * Cs * dir*bz(i,0)/modB * by(i,0)/max(abs(by(i,0).1.d-8)

		p(i,0)=p(i,1)
		p(i,0) = max(pmin,p(i,0))

		!!$     boundary conditions on the top boundary (sonic outflow)

		modB = sqrt(bx(i,ny+1)**2 + by(i,ny+1)**2 + bz(i,ny+1)**2)
		Cs = sqrt(rsh*p(i,ny)/rho(i,ny))
		dir = sign(1.d0,by(i,ny+1))

		rho(i,ny+1) = max(rho(i,ny),rhomin)

		u(i,ny+1) = Mach_boundary * Cs * dir*bx(i,ny+1)/modB
		v(i,ny+1) = Mach_boundary * Cs * dir*by(i,ny+1)/modB
		w(i,ny+1) = Mach_boundary * Cs * dir*bz(i,ny+1)/modB

!		u(i,ny+1) = Mach_boundary * Cs * dir*bx(i,ny+1)/modB * by(i,0)/max(abs(by(i,0).1.d-8)
!		v(i,ny+1) = Mach_boundary * Cs * dir*by(i,ny+1)/modB
!		w(i,ny+1) = Mach_boundary * Cs * dir*bz(i,ny+1)/modB

		p(i,ny+1)=p(i,ny)
		p(i,ny+1) = max(pmin,p(i,ny+1))


	enddo

	return

end subroutine boundary_vacuum_CS


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine  boundary_vacuum_CS_2(u,v,w,rho,p,bx,by,bz)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! sonic outflow at the edge

	use mod_parameters, ONLY: nx, ny, dx, rsh, rmu0, torus, pmin, rhomin
	use mod_arrays, only : nmx, nmy, Rmaj

	real(kind=dkind), dimension(0:nmx+1,0:nmy+1) :: rho, u, v, w, p, bx, by, bz
	real(kind=dkind) :: modB, dir
	real(kind=dkind), save :: Cs = 0.d0
	integer :: i, j

	if(Cs==0.d0) Cs = sqrt(rsh*p_boundary/rho_boundary)

	do j=1,ny

		!!$   boundary conditions on left boundary (sonic outflow)

		modB = sqrt(bx(0,j)**2 + by(0,j)**2 + bz(0,j)**2)
		dir = -sign(1.d0,bx(0,j))

		rho(0,j) = max(rho(1,j),rhomin)

		u(0,j) = Mach_boundary * Cs * dir*bx(0,j)/modB
		v(0,j) = Mach_boundary * Cs * dir*by(0,j)/modB
		w(0,j) = Mach_boundary * Cs * dir*bz(0,j)/modB

		p(0,j)=p(1,j)+(by(1,j)**2-by(0,j)**2)/  &
			(2.d0*rmu0)-torus*((rho(1,j)*(w(1,j)**2+w(0,j)**2)  &
			)*dx/(rmaj(1)+rmaj(0)))
		p(0,j) = max(pmin,p(0,j))

		!!$   boundary conditions on right boundary (sonic outflow)

		modB = sqrt(bx(nx+1,j)**2 + by(nx+1,j)**2 + bz(nx+1,j)**2)
		dir = sign(1.d0,bx(nx+1,j))

		rho(nx+1,j) = max(rho(nx,j),rhomin)

		u(nx+1,j) = Mach_boundary * Cs * dir*bx(nx+1,j)/modB
		v(nx+1,j) = Mach_boundary * Cs * dir*by(nx+1,j)/modB
		w(nx+1,j) = Mach_boundary * Cs * dir*bz(nx+1,j)/modB

		p(nx+1,j)=p(nx,j)+(by(nx,j)**2-  &
		    by(nx+1,j)**2)/(2.*rmu0)+torus*(  &
		   (rho(nx,j)*(w(nx,j)**2+w(nx+1,j)**2)  &
		    )*dx/(rmaj(nx)+rmaj(nx+1)))
		p(nx+1,j) = max(pmin,p(nx+1,j))

	enddo




	do i=1,nx

		!!$     boundary conditions on the bottom boundary (sonic outflow)

		modB = sqrt(bx(i,0)**2 + by(i,0)**2 + bz(i,0)**2)
		dir = -sign(1.d0,by(i,0))

		rho(i,0) = max(rho(i,1),rhomin)

		u(i,0) = Mach_boundary * Cs * dir*bx(i,0)/modB
		v(i,0) = Mach_boundary * Cs * dir*by(i,0)/modB
		w(i,0) = Mach_boundary * Cs * dir*bz(i,0)/modB

		p(i,0)=p(i,1)
		p(i,0) = max(pmin,p(i,0))

		!!$     boundary conditions on the top boundary (sonic outflow)

		modB = sqrt(bx(i,ny+1)**2 + by(i,ny+1)**2 + bz(i,ny+1)**2)
		dir = sign(1.d0,by(i,ny+1))

		rho(i,ny+1) = max(rho(i,ny),rhomin)

		u(i,ny+1) = Mach_boundary * Cs * dir*bx(i,ny+1)/modB
		v(i,ny+1) = Mach_boundary * Cs * dir*by(i,ny+1)/modB
		w(i,ny+1) = Mach_boundary * Cs * dir*bz(i,ny+1)/modB

		p(i,ny+1)=p(i,ny)
		p(i,ny+1) = max(pmin,p(i,ny+1))


	enddo

	return

end subroutine boundary_vacuum_CS_2


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine  boundary_vacuum_CS_3(u,v,w,rho,p,bx,by,bz,eta_phys)
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
! sonic outflow at the edge

	use mod_parameters, ONLY: nx, ny, dx, dy, rsh, rmu0, torus, pmin, rhomin, xlength, ylength
	use mod_arrays, only : nmx, nmy, Rmaj
	use vacuum_module, only : eta_phys_boundary

	real(kind=dkind), dimension(0:nmx+1,0:nmy+1) :: rho, u, v, w, p, bx, by, bz, eta_phys
	real(kind=dkind) :: modB, dir
	real(kind=dkind), save :: Cs = 0.d0
	real(kind=dkind), save :: vtheta = 0.d0
	integer :: i, j

	if(Cs==0.d0) Cs = sqrt(rsh*p_boundary/rho_boundary)

	do j=1,ny

		!!$   boundary conditions on left boundary (sonic outflow)

		modB = sqrt(bx(0,j)**2 + by(0,j)**2 + bz(0,j)**2)
		dir = -sign(1.d0,bx(0,j))

		rho(0,j) = max(rho(1,j),rhomin)
!		eta_phys(0,j) = max(eta_phys(1,j),eta_phys_boundary)
		eta_phys(0,j) = eta_phys_boundary

		u(0,j) = Mach_boundary * Cs * dir*bx(0,j)/modB
		v(0,j) = Mach_boundary * Cs * dir*by(0,j)/modB
		w(0,j) = Mach_boundary * Cs * dir*bz(0,j)/modB

		p(0,j)=p(1,j)
		p(0,j) = max(pmin,p(0,j))

		!!$   boundary conditions on right boundary (sonic outflow)

		modB = sqrt(bx(nx+1,j)**2 + by(nx+1,j)**2 + bz(nx+1,j)**2)
		dir = sign(1.d0,bx(nx+1,j))

		rho(nx+1,j) = max(rho(nx,j),rhomin)
!		eta_phys(nx+1,j) = max(eta_phys(nx,j),eta_phys_boundary)
		eta_phys(nx+1,j) = eta_phys_boundary

		u(nx+1,j) = Mach_boundary * Cs * dir*bx(nx+1,j)/modB
		v(nx+1,j) = Mach_boundary * Cs * dir*by(nx+1,j)/modB
		w(nx+1,j) = Mach_boundary * Cs * dir*bz(nx+1,j)/modB

		p(nx+1,j)=p(nx,j)
		p(nx+1,j) = max(pmin,p(nx+1,j))

	enddo




	do i=1,nx

		!!$     boundary conditions on the bottom boundary (sonic outflow)

		modB = sqrt(bx(i,0)**2 + by(i,0)**2 + bz(i,0)**2)
		dir = -sign(1.d0,by(i,0))

		rho(i,0) = max(rho(i,1),rhomin)
!		eta_phys(i,0) = max(eta_phys(i,1),eta_phys_boundary)
		eta_phys(i,0) = eta_phys_boundary

		u(i,0) = Mach_boundary * Cs * dir*bx(i,0)/modB
		v(i,0) = Mach_boundary * Cs * dir*by(i,0)/modB
		w(i,0) = Mach_boundary * Cs * dir*bz(i,0)/modB

		p(i,0)=p(i,1)
		p(i,0) = max(pmin,p(i,0))

		!!$     boundary conditions on the top boundary (sonic outflow)

		modB = sqrt(bx(i,ny+1)**2 + by(i,ny+1)**2 + bz(i,ny+1)**2)
		dir = sign(1.d0,by(i,ny+1))

		rho(i,ny+1) = max(rho(i,ny),rhomin)
!		eta_phys(i,ny+1) = max(eta_phys(i,ny),eta_phys_boundary)
		eta_phys(i,ny+1) = eta_phys_boundary

		u(i,ny+1) = Mach_boundary * Cs * dir*bx(i,ny+1)/modB
		v(i,ny+1) = Mach_boundary * Cs * dir*by(i,ny+1)/modB
		w(i,ny+1) = Mach_boundary * Cs * dir*bz(i,ny+1)/modB

		p(i,ny+1)=p(i,ny)
		p(i,ny+1) = max(pmin,p(i,ny+1))


	enddo

!!$	if(vtheta==0.d0) then
!!$
!!$		do i = 1,nx
!!$
!!$			vtheta = vtheta + dx*u(i,ny+1)
!!$			vtheta = vtheta - dx*u(i,0)
!!$
!!$		enddo
!!$
!!$		do j = 1,ny
!!$
!!$			vtheta = vtheta + dy*v(nx+1,j)
!!$			vtheta = vtheta - dy*v(0,j)
!!$
!!$		enddo
!!$
!!$		vtheta = vtheta / (2.d0*(xlength+ylength))
!!$
!!$		open(33, file='vtheta_source.dat')
!!$		write(33,*) vtheta, Cs
!!$		close(33)
!!$
!!$	endif

	return

end subroutine boundary_vacuum_CS_3


!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
subroutine boundary_cleanup
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	deallocate(sort_grid)

end subroutine boundary_cleanup


end module  boundary_routines

